/* packet-h261.c
 *
 * Routines for ITU-T Recommendation H.261 dissection
 *
 * Copyright 2000, Philips Electronics N.V.
 * Andreas Sikkema <h323@ramdyne.nl>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/*
 * This dissector tries to dissect the H.261 protocol according to Annex C
 * of ITU-T Recommendation H.225.0 (02/98)
 */


#include "config.h"

#include <epan/packet.h>

#include <epan/rtp_pt.h>
#include <epan/iax2_codec_type.h>

void proto_register_h261(void);
void proto_reg_handoff_h261(void);

static dissector_handle_t h261_handle;

/* H.261 header fields             */
static int proto_h261;
static int hf_h261_sbit;
static int hf_h261_ebit;
static int hf_h261_ibit;
static int hf_h261_vbit;
static int hf_h261_gobn;
static int hf_h261_mbap;
static int hf_h261_quant;
static int hf_h261_hmvd; /* Mislabeled in a figure in section C.3.1 as HMDV */
static int hf_h261_vmvd;
static int hf_h261_data;

/* H.261 fields defining a sub tree */
static gint ett_h261;

static int
dissect_h261( tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_ )
{
	proto_item *ti            = NULL;
	proto_tree *h261_tree     = NULL;
	unsigned int offset       = 0;
	static int * const bits[] = {
		/* SBIT 1st octet, 3 bits */
		&hf_h261_sbit,
		/* EBIT 1st octet, 3 bits */
		&hf_h261_ebit,
		/* I flag, 1 bit */
		&hf_h261_ibit,
		/* V flag, 1 bit */
		&hf_h261_vbit,
		NULL
	};

	col_set_str(pinfo->cinfo, COL_PROTOCOL, "H.261");

	col_set_str(pinfo->cinfo, COL_INFO, "H.261 message");

	ti = proto_tree_add_item( tree, proto_h261, tvb, offset, -1, ENC_NA);
	h261_tree = proto_item_add_subtree( ti, ett_h261 );

	proto_tree_add_bitmask_list(h261_tree, tvb, offset, 1, bits, ENC_NA);
	offset++;

	/* GOBN 2nd octet, 4 bits */
	proto_tree_add_item( h261_tree, hf_h261_gobn, tvb, offset, 1, ENC_NA);
	/* MBAP 2nd octet, 4 bits, 3rd octet 1 bit */
	proto_tree_add_item( h261_tree, hf_h261_mbap, tvb, offset, 2, ENC_BIG_ENDIAN);
	offset++;

	/* QUANT 3rd octet, 5 bits (starting at bit 2!) */
	proto_tree_add_item( h261_tree, hf_h261_quant, tvb, offset, 1, ENC_NA);

	/* HMVD 3rd octet 2 bits, 4th octet 3 bits */
	proto_tree_add_item( h261_tree, hf_h261_hmvd, tvb, offset, 2, ENC_BIG_ENDIAN);
	offset++;

	/* VMVD 4th octet, last 5 bits */
	proto_tree_add_item( h261_tree, hf_h261_vmvd, tvb, offset, 1, ENC_NA);
	offset++;

	/* The rest of the packet is the H.261 stream */
	proto_tree_add_item( h261_tree, hf_h261_data, tvb, offset, -1, ENC_NA);

	return tvb_captured_length(tvb);
}

void
proto_register_h261(void)
{
	static hf_register_info hf[] =
	{
		{
			&hf_h261_sbit,
			{
				"Start bit position",
				"h261.sbit",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0xe0,
				NULL, HFILL
			}
		},
		{
			&hf_h261_ebit,
			{
				"End bit position",
				"h261.ebit",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x1c,
				NULL, HFILL
			}
		},
		{
			&hf_h261_ibit,
			{
				"Intra frame encoded data flag",
				"h261.i",
				FT_BOOLEAN,
				8,
				NULL,
				0x02,
				NULL, HFILL
			}
		},
		{
			&hf_h261_vbit,
			{
				"Motion vector flag",
				"h261.v",
				FT_BOOLEAN,
				8,
				NULL,
				0x01,
				NULL, HFILL
			}
		},
		{
			&hf_h261_gobn,
			{
				"GOB Number",
				"h261.gobn",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0xF0,
				NULL, HFILL
			}
		},
		{
			&hf_h261_mbap,
			{
				"Macroblock address predictor",
				"h261.mbap",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x0F80,
				NULL, HFILL
			}
		},
		{
			&hf_h261_quant,
			{
				"Quantizer",
				"h261.quant",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x7C,
				NULL, HFILL
			}
		},
		{
			&hf_h261_hmvd,
			{
				"Horizontal motion vector data",
				"h261.hmvd",
				FT_UINT16,
				BASE_DEC,
				NULL,
				0x03E0,
				NULL, HFILL
			}
		},
		{
			&hf_h261_vmvd,
			{
				"Vertical motion vector data",
				"h261.vmvd",
				FT_UINT8,
				BASE_DEC,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
		{
			&hf_h261_data,
			{
				"H.261 stream",
				"h261.stream",
				FT_BYTES,
				BASE_NONE,
				NULL,
				0x0,
				NULL, HFILL
			}
		},
};

	static gint *ett[] =
	{
		&ett_h261,
	};


	proto_h261 = proto_register_protocol("ITU-T Recommendation H.261",
	    "H.261", "h261");
	proto_register_field_array(proto_h261, hf, array_length(hf));
	proto_register_subtree_array(ett, array_length(ett));

	h261_handle = register_dissector("h261", dissect_h261, proto_h261);
}

void
proto_reg_handoff_h261(void)
{
	dissector_add_uint("rtp.pt", PT_H261, h261_handle);
	dissector_add_uint("iax2.codec", AST_FORMAT_H261, h261_handle);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: t
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 noexpandtab:
 * :indentSize=8:tabSize=8:noTabs=false:
 */
