import os
import shlex

import lldb
from jb_lldb_command_base import LLDBRiderCppCommand


class JBGetSymFiles(LLDBRiderCppCommand):
    command: str = 'jb_get_module_symbols_files'

    def __init__(self, debugger: lldb.SBDebugger, bindings: dict):
        super().__init__(debugger, bindings)

    def get_short_help(self) -> str:
        return "Get the symbols files for the selected target. Usage: jb_get_module_symbols_files module1 [moduleN]"

    def __call__(self, debugger: lldb.SBDebugger, args: str,
                 execution_context: lldb.SBExecutionContext,
                 command_return: lldb.SBCommandReturnObject) -> None:

        if len(args) == 0:
            command_return.SetStatus(lldb.eReturnStatusFailed)
            command_return.SetError('Expected at least one module')
            return

        modules: list[str] = shlex.split(args)
        self.get_sym_files(debugger, modules, command_return)

    def get_sym_files(self,
                      debugger: lldb.SBDebugger, modules: list[str],
                      command_return: lldb.SBCommandReturnObject) -> None:
        self._log('Getting symbols files: enter')
        try:
            target: lldb.SBTarget = debugger.GetSelectedTarget()
            assert target and target.IsValid(), "No target selected"

            for module_name in modules:
                self._process_module(command_return, module_name, target)
        except Exception as e:
            command_return.SetStatus(lldb.eReturnStatusFailed)
            command_return.SetError('Error on getting symbols files: ' + str(e))
            return None

        command_return.SetStatus(lldb.eReturnStatusSuccessFinishResult)
        self._log('Getting symbols files: done')

    @staticmethod
    def _process_module(
            command_return: lldb.SBCommandReturnObject,
            module_name: str,
            target: lldb.SBTarget) -> None:

        module: lldb.SBModule = target.FindModule(lldb.SBFileSpec(module_name))
        if not module:
            command_return.AppendMessage(f"{module_name}=ERROR:No module found")
            return None

        bin_file: lldb.SBFileSpec = module.GetFileSpec()
        symbol_file: lldb.SBFileSpec = module.GetSymbolFileSpec()
        if not symbol_file.IsValid():
            command_return.AppendMessage(f"{module_name}=ERROR:No symbol file found")
            return None

        if symbol_file.fullpath == bin_file.fullpath:
            command_return.AppendMessage(f"{module_name}=NONE")
            return None

        command_return.AppendMessage(f"{module_name}=FILE:{os.path.abspath(symbol_file.fullpath)}")
        return None


def __lldb_init_module(debugger: lldb.SBDebugger, _):
    JBGetSymFiles.register_lldb_command(debugger, __name__)
