import abc
from platform import system

import lldb

if system() == "Windows":
    from renderers import jb_lldb_logging as logging

class LLDBRiderCppCommand(abc.ABC):
    """
    See `CommandObjectType` <https://lldb.llvm.org/use/python-reference.html>.
    """

    command: str

    @abc.abstractmethod
    def __init__(self, _: lldb.SBDebugger, __: dict):
        pass

    @abc.abstractmethod
    def get_short_help(self) -> str:
        pass

    @classmethod
    def register_lldb_command(cls, debugger: lldb.SBDebugger, module_name: str) -> None:
        command = f'command script add --class {module_name}.{cls.__name__} {cls.command}'
        debugger.HandleCommand(command)

    @abc.abstractmethod
    def __call__(self, debugger: lldb.SBDebugger, args: str,
                 execution_context: lldb.SBExecutionContext,
                 command_return: lldb.SBCommandReturnObject) -> None:
        pass

    @classmethod
    def _log(cls, fmt: str, *args, **kwargs) -> None:
        if system() == "Windows":
            logging.log(f"{cls.__name__}> {fmt}", *args, **kwargs)
        else:
            print(f"{cls.__name__}> {fmt}", *args, **kwargs)

