"use strict";
/**
 * The plugin is intended to override and/or extend some behaviour of ts service,
 * please before implementing custom logic here, first consider to implement logic based on tsserver standard api.
 */
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var get_element_type_ts_server_1 = require("./get-element-type-ts-server");
var openExternalProjectExCommand = "ideOpenExternalProject";
var projectInfoExCommand = "ideProjectInfo";
var emitFileExCommand = "ideCompileOnSaveEmitFile";
var getElementTypeCommand = "ideGetElementType";
var getSymbolTypeCommand = "ideGetSymbolType";
var getTypePropertiesCommand = "ideGetTypeProperties";
var addedCommands = false;
var init = (function (modules) {
    var ts = modules.typescript;
    var tsVersion = parseNumbersInVersion(ts.version);
    function addNewHandlers(info) {
        var projectService = info.project.projectService;
        projectService.logger.info("Called handler processing");
        var session = info.session;
        if (session == undefined) {
            projectService.logger.info("There is no session in info.");
            return;
        }
        if (session.addProtocolHandler == undefined) {
            // addProtocolHandler was introduced in TS 4.4 or 4.5 in 2021, see https://github.com/microsoft/TypeScript/issues/43893
            projectService.logger.info("There is no addProtocolHandler method.");
            return;
        }
        if (!isVersionMoreOrEqual(tsVersion, 5)) {
            projectService.logger.info("Skip adding handlers for old version: " + ts.version);
            return;
        }
        session.addProtocolHandler(openExternalProjectExCommand, openExternalProjectHandler.bind(null, ts, projectService));
        session.addProtocolHandler(projectInfoExCommand, projectInfoHandler.bind(null, ts, projectService));
        session.addProtocolHandler(emitFileExCommand, emitFileHandler.bind(null, ts, projectService));
        session.addProtocolHandler(getElementTypeCommand, getElementTypeCommandHandler.bind(null, ts, projectService));
        session.addProtocolHandler(getSymbolTypeCommand, getSymbolTypeCommandHandler.bind(null, ts, projectService));
        session.addProtocolHandler(getTypePropertiesCommand, getTypePropertiesCommandHandler.bind(null, ts, projectService));
        projectService.logger.info("IDE specific commands are successfully added.");
    }
    function create(info) {
        if (!addedCommands) {
            addNewHandlers(info);
            addedCommands = true;
        }
        (0, get_element_type_ts_server_1.decorateLanguageService)(info.languageService);
        return info.languageService;
    }
    return { create: create };
});
function openExternalProjectEx(ts, projectService, externalProject) {
    var fileName = externalProject.projectFileName;
    if (fileName == null)
        return; //nothing to open
    var tsConfigPath = ts.server.toNormalizedPath(fileName);
    /*
     * There's no public API method to open a project with the given config file.
     * The projectService.openExternalProject() method doesn't take the config file as a
     * parameter; instead, it tries to infer the config file, allowing only tsconfig.json
     * and jsconfig.json names. Therefore, we use this method only when the name matches.
     * Otherwise, we have to use private methods.
     */
    if (tsConfigPath.endsWith("/tsconfig.json") ||
        tsConfigPath.endsWith("/jsconfig.json") ||
        !tsConfigPath.endsWith(".json")) {
        projectService.openExternalProject(externalProject);
        return;
    }
    var _projectService = projectService;
    if (_projectService.createAndLoadConfiguredProject && _projectService.externalProjectToConfiguredProjectMap) {
        _projectService.externalProjectToConfiguredProjectMap.set(externalProject.projectFileName, [tsConfigPath]);
        var project = _projectService.createAndLoadConfiguredProject(tsConfigPath, "Creating own configured project in external project: ".concat(externalProject.projectFileName));
        project.updateGraph();
        if (project.addExternalProjectReference) {
            project.addExternalProjectReference();
        }
        return;
    }
    if (_projectService.createConfiguredProject && _projectService.externalProjectToConfiguredProjectMap) {
        // Like in ProjectService.openExternalProject()
        var project = _projectService.createConfiguredProject(tsConfigPath, "Creating configured project in external project: ".concat(externalProject.projectFileName));
        if (!projectService.getHostPreferences().lazyConfiguredProjectsFromExternalProject) {
            project.updateGraph();
        }
        _projectService.externalProjectToConfiguredProjectMap.set(externalProject.projectFileName, new Set([project]));
        return;
    }
    // TypeScript will anyway try to infer the project on the first 'open'. However, we throw to appear in logs.
    throw new Error("Could not create configured project with tsConfigPath=".concat(tsConfigPath));
}
var getElementTypeCommandHandler = function (ts, projectService, request) {
    return (0, get_element_type_ts_server_1.getElementTypeTsServer)(ts, projectService, request) || emptyDoneResponse();
};
var getSymbolTypeCommandHandler = function (ts, projectService, request) {
    return (0, get_element_type_ts_server_1.getSymbolTypeTsServer)(ts, projectService, request) || emptyDoneResponse();
};
var getTypePropertiesCommandHandler = function (ts, projectService, request) {
    return (0, get_element_type_ts_server_1.getTypePropertiesTsServer)(ts, projectService, request) || emptyDoneResponse();
};
var openExternalProjectHandler = function (ts, projectService, request) {
    var openRequest = request;
    var externalProject = openRequest.arguments;
    openExternalProjectEx(ts, projectService, externalProject);
    return {
        response: true,
        responseRequired: true
    };
};
var emitFileHandler = function (ts, projectService, request) {
    var host = projectService.host;
    var compileRequest = request;
    var args = compileRequest.arguments;
    var file = args.file ? ts.server.toNormalizedPath(args.file) : undefined;
    var projectFileName = args.projectFileName;
    if (file == null) {
        projectService.logger.info("There is no file to compile: " + projectFileName);
        return {
            response: {
                emitSkipped: true,
                diagnostics: []
            },
            responseRequired: true
        };
    }
    var diagnostics = [];
    var generatedFiles = [];
    var processedFiles = [];
    var result = {
        emitSkipped: false,
        generatedFiles: generatedFiles,
        processedFiles: processedFiles,
        diagnostics: [
            {
                file: file,
                diagnostics: diagnostics
            }
        ]
    };
    var project = projectFileName ?
        projectService.findProject(projectFileName) :
        projectService.getDefaultProjectForFile(file, false);
    var openedProject = false;
    if (project == null) {
        var fileToOpen = projectFileName ? projectFileName : file;
        openExternalProjectEx(ts, projectService, {
            options: {},
            projectFileName: fileToOpen,
            rootFiles: [{ fileName: fileToOpen }]
        });
        openedProject = true;
        project = projectService.findProject(fileToOpen);
    }
    if (project != null) {
        var scriptInfo = project.getScriptInfo(file);
        if (scriptInfo) {
            var result_1 = project.emitFile(scriptInfo, function (path, data, writeByteOrderMark) {
                generatedFiles.push(path);
                return host.writeFile(path, data, writeByteOrderMark);
            });
            result_1.emitSkipped = result_1.emitSkipped;
            if (args.richResponse) {
                var resultDiagnostics = [];
                var languageService = project.getLanguageService();
                resultDiagnostics.push.apply(resultDiagnostics, languageService.getSemanticDiagnostics(file));
                resultDiagnostics.push.apply(resultDiagnostics, languageService.getSyntacticDiagnostics(file));
                resultDiagnostics.push.apply(resultDiagnostics, result_1.diagnostics);
                diagnostics.push.apply(diagnostics, resultDiagnostics.map(function (el) { return formatDiagnosticToProtocol(ts, el, true); }));
            }
            if (result_1.emitSkipped) {
                //no emitted -> reset info
                generatedFiles.length = 0;
            }
        }
        else {
            projectService.logger.info("There is no script info. File ".concat(file, ", project ").concat(projectFileName));
        }
    }
    else {
        projectService.logger.info("There is no project. File ".concat(file, ", project ").concat(projectFileName));
    }
    if (openedProject) {
        //do not close for perf reason
        // projectService.closeExternalProject(projectFileName);
    }
    return {
        response: result,
        responseRequired: true
    };
};
var projectInfoHandler = function (_ts, projectService, _request) {
    var infos = [];
    var configuredProjects = projectService.configuredProjects;
    configuredProjects.forEach(function (configuredProject) {
        addProjectInfo(configuredProject, infos);
    });
    for (var _i = 0, _a = projectService.inferredProjects; _i < _a.length; _i++) {
        var inferredProject = _a[_i];
        addProjectInfo(inferredProject, infos);
    }
    for (var _b = 0, _c = projectService.externalProjects; _b < _c.length; _b++) {
        var externalProject = _c[_b];
        addProjectInfo(externalProject, infos);
    }
    return {
        responseRequired: true,
        response: infos
    };
};
function addProjectInfo(project, infos) {
    var name = project.getProjectName();
    var regularFileInfos = project.getFileNames(false).map(function (el) {
        var info = project.getScriptInfo(el);
        return {
            fileName: el,
            isOpen: !!info && info.isScriptOpen(),
            isExternal: false
        };
    });
    var externalFileInfos = [];
    if (project.getExternalFiles) {
        externalFileInfos = project.getExternalFiles().map(function (el) {
            var info = project.getScriptInfo(el);
            return {
                fileName: el,
                isOpen: !!info && info.isScriptOpen(),
                isExternal: true
            };
        });
    }
    infos.push({
        projectName: name,
        fileInfos: regularFileInfos.concat(externalFileInfos)
    });
}
/** also update tsLanguageService/typescript/util.ts */
function parseNumbersInVersion(version) {
    var result = [];
    var versions = version.split(/[.-]/);
    for (var _i = 0, versions_1 = versions; _i < versions_1.length; _i++) {
        version = versions_1[_i];
        if (!version)
            break;
        var currentNumber = Number(version);
        if (currentNumber == null || isNaN(currentNumber))
            break;
        result = result.concat(currentNumber);
    }
    return result;
}
function diagnosticCategoryName(ts, d, lowerCase) {
    if (lowerCase === void 0) { lowerCase = true; }
    var name = ts.DiagnosticCategory[d.category];
    return lowerCase ? name.toLowerCase() : name;
}
function isVersionMoreOrEqual(version) {
    var expected = [];
    for (var _i = 1; _i < arguments.length; _i++) {
        expected[_i - 1] = arguments[_i];
    }
    for (var i = 0; i < expected.length; i++) {
        var expectedNumber = expected[i];
        var currentNumber = version.length > i ? version[i] : 0;
        if (currentNumber < expectedNumber)
            return false;
        if (currentNumber > expectedNumber)
            return true;
    }
    return version.length >= expected.length;
}
function formatDiagnosticToProtocol(ts, diag, includeFileName) {
    var start = (diag.file && convertToLocation(ts.getLineAndCharacterOfPosition(diag.file, diag.start)));
    var end = (diag.file && convertToLocation(ts.getLineAndCharacterOfPosition(diag.file, diag.start + diag.length)));
    var text = ts.flattenDiagnosticMessageText(diag.messageText, "\n");
    var code = diag.code, source = diag.source;
    var category = diagnosticCategoryName(ts, diag);
    var common = {
        start: start,
        end: end,
        text: text,
        code: code,
        category: category,
        reportsUnnecessary: diag.reportsUnnecessary,
        reportsDeprecated: diag.reportsDeprecated,
        source: source
    };
    return includeFileName
        ? __assign(__assign({}, common), { fileName: diag.file && diag.file.fileName }) : common;
}
function emptyDoneResponse() {
    return {
        responseRequired: true,
        response: null
    };
}
function convertToLocation(lineAndCharacter) {
    return {
        line: lineAndCharacter.line + 1,
        offset: lineAndCharacter.character + 1
    };
}
module.exports = init;
//# sourceMappingURL=index.js.map