import re
from typing import Tuple, Optional


class CppParser:
    _REGEX_IDENT = r"[A-Za-z_$][\w$]*"
    _REGEX_SUBSCRIPT = r"\[\d+]"

    # Expression like that: foo->bar[1][0].baz
    _PATTERN_TRIVIAL_EXPR = re.compile(fr"^{_REGEX_IDENT}(:?{_REGEX_SUBSCRIPT})*"
                                       fr"(:?(:?\.|->){_REGEX_IDENT}(:?{_REGEX_SUBSCRIPT})*)*$")
    _PATTERN_ARRAY_ACCESS = re.compile(f"^{_REGEX_SUBSCRIPT}$")
    _PATTERN_LAMBDA = re.compile(r"^.*<lambda_[0-9a-f]{32}>.*$")

    @classmethod
    def is_trivial_expression(cls, expr: str) -> bool:
        return cls._PATTERN_TRIVIAL_EXPR.match(expr) is not None

    @classmethod
    def is_array_access_expr(cls, expr: str) -> bool:
        return cls._PATTERN_ARRAY_ACCESS.match(expr) is not None

    @classmethod
    def has_lambda_in_type_expr(cls, type_expr: str) -> bool:
        return cls._PATTERN_LAMBDA.match(type_expr) is not None

    @staticmethod
    def remove_cpp_comments(code: str) -> str:
        comment_pos = []
        in_string = False
        i = 0
        code_len = len(code)
        while i < code_len:
            char = code[i]
            if not in_string and char == "/" and i + 1 < code_len:
                comment_end_seq = None
                next_char = code[i + 1]
                if next_char == "/":
                    comment_end_seq = "\n"
                elif next_char == "*":
                    comment_end_seq = "*/"
                if comment_end_seq is not None:
                    comment_end = code.find(comment_end_seq, i + 2)
                    comment_end = code_len if comment_end < 0 else comment_end + len(comment_end_seq)
                    comment_pos.append((i, comment_end))
                    i = comment_end
                    continue

            if char == '"':
                in_string = not in_string

            i += 1

        if not comment_pos:
            return code

        code_no_comments = []
        code_pos = 0
        for (comment_start, comment_end) in comment_pos:
            if code_pos < comment_start:
                code_no_comments.append(code[code_pos:comment_start].strip())
            code_pos = comment_end
        if code_pos < code_len:
            code_no_comments.append(code[code_pos:].strip())

        return ''.join(code_no_comments)

    @staticmethod
    def is_outer_parentheses_balanced(s: str) -> bool:
        if not s.startswith('(') or not s.endswith(')'):
            return False
        counter = 0
        in_string = False
        for i in range(1, len(s) - 1):
            char = s[i]
            if char == '"':
                in_string = not in_string
            elif not in_string:
                if char == '(':
                    counter += 1
                elif char == ')':
                    counter -= 1
                if counter < 0:
                    # There was a closing parenthesis without a matching opening one
                    return False
        return counter == 0 and not in_string

    @classmethod
    def try_remove_outer_parentheses(cls, s: str) -> str:
        s = s.strip()
        while cls.is_outer_parentheses_balanced(s):
            s = s[1:-1].strip()
        return s.strip()

    @classmethod
    def try_merge_deref_and_address_of(cls, s: str) -> str:
        if not s.startswith("(*(&(") or not s.endswith(")))"):
            return s

        if not cls.is_outer_parentheses_balanced(s):
            return s
        # (*(&(<expr>))) -> (&(<expr>))
        no_deref = s[2:-1]

        if not cls.is_outer_parentheses_balanced(no_deref):
            return s
        # (&(<expr>)) -> (<expr>)
        no_address_of = no_deref[2:-1]

        if not cls.is_outer_parentheses_balanced(no_address_of):
            return s
        return no_address_of

    @classmethod
    def cut_deref_or_address_of_from_trivial_expression(cls, expr: str) -> Tuple[Optional[str], Optional[str]]:
        expr = expr.strip()
        if expr.startswith("*") or expr.startswith("&"):
            specifier = expr[0]
            sub_expr = CppParser.try_remove_outer_parentheses(expr[1:])
            if cls.is_trivial_expression(sub_expr):
                return specifier, sub_expr
        return None, None

    @classmethod
    def simplify_cpp_expression(cls, expr: str) -> str:
        expr = cls.remove_cpp_comments(expr.strip())
        return cls.try_remove_outer_parentheses(expr)
