/*
 * This file is part of PokéFinder
 * Copyright (C) 2017-2024 by Admiral_Fish, bumba, and EzPzStreamz
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "CheckList.hpp"
#include <QApplication>
#include <QLineEdit>
#include <QListView>
#include <QMouseEvent>
#include <QStandardItemModel>
#include <algorithm>

CheckList::CheckList(QWidget *parent) : QComboBox(parent), model(new QStandardItemModel(this))
{
    setModel(model);

    setEditable(true);
    lineEdit()->setReadOnly(true);
    lineEdit()->installEventFilter(this);

    connect(lineEdit(), &QLineEdit::selectionChanged, lineEdit(), &QLineEdit::deselect);
    connect(qobject_cast<QListView *>(view()), &QAbstractItemView::pressed, this, &CheckList::itemPressed);
    connect(model, &QAbstractItemModel::dataChanged, this, &CheckList::modelDataChanged);
}

std::vector<bool> CheckList::getChecked() const
{
    std::vector<bool> result;
    if (checkState() == Qt::PartiallyChecked)
    {
        for (int i = 0; i < model->rowCount(); i++)
        {
            result.emplace_back(model->item(i)->checkState() == Qt::Checked);
        }
    }
    else
    {
        result = std::vector<bool>(model->rowCount(), true);
    }
    return result;
}

std::vector<u16> CheckList::getCheckedData() const
{
    auto checked = getChecked();
    std::vector<u16> data;
    for (int i = 0; i < checked.size(); i++)
    {
        if (checked[i])
        {
            data.emplace_back(itemData(i).toUInt());
        }
    }
    return data;
}

void CheckList::resetChecks()
{
    for (auto i = 0; i < model->rowCount(); i++)
    {
        model->item(i)->setCheckState(Qt::Unchecked);
    }
}

void CheckList::setChecks(const std::vector<bool> &flags)
{
    for (size_t i = 0; i < flags.size() && i < model->rowCount(); i++)
    {
        model->item(i)->setCheckState(flags[i] ? Qt::Checked : Qt::Unchecked);
    }
}

void CheckList::setup(const std::vector<std::string> &items)
{
    if (!items.empty())
    {
        clear();
        for (const auto &item : items)
        {
            addItem(QString::fromStdString(item));
        }
    }

    setupChecks();
}

void CheckList::setup(const std::vector<std::string> &items, const std::vector<u16> &data)
{
    assert(items.size() == data.size());
    if (!items.empty())
    {
        clear();
        for (int i = 0; i < items.size(); i++)
        {
            addItem(QString::fromStdString(items[i]), data[i]);
        }
    }

    setupChecks();
}

bool CheckList::eventFilter(QObject *object, QEvent *event)
{
    if (object == lineEdit() && event->type() == QEvent::MouseButtonPress)
    {
        auto *mouse = reinterpret_cast<QMouseEvent *>(event);
        if (mouse->modifiers() == Qt::ControlModifier)
        {
            resetChecks();
        }
        else
        {
            showPopup();
        }
        return true;
    }

    return false;
}

Qt::CheckState CheckList::checkState() const
{
    int total = model->rowCount();
    int checked = 0;
    int unchecked = 0;

    for (int i = 0; i < total; i++)
    {
        if (model->item(i)->checkState() == Qt::Checked)
        {
            checked++;
        }
        else if (model->item(i)->checkState() == Qt::Unchecked)
        {
            unchecked++;
        }
    }

    return checked == total ? Qt::Checked : unchecked == total ? Qt::Unchecked : Qt::PartiallyChecked;
}

void CheckList::setupChecks()
{
    auto font = fontMetrics();
    int width = 0;
    for (int i = 0; i < model->rowCount(); i++)
    {
        QStandardItem *item = model->item(i);
        item->setCheckState(Qt::Unchecked);
        item->setFlags(Qt::ItemIsUserCheckable | Qt::ItemIsEnabled);

        width = std::max(width, font.size(0, item->text()).width());
    }

    width += 2 * font.size(0, " ").width() + iconSize().width() + QApplication::style()->pixelMetric(QStyle::PM_ScrollBarExtent);
    width *= 1.25f;

    view()->setMinimumWidth(width);
}

void CheckList::modelDataChanged()
{
    QString text;

    switch (checkState())
    {
    case Qt::Checked:
    case Qt::Unchecked:
        text = tr("Any");
        break;
    case Qt::PartiallyChecked:
        for (int i = 0; i < model->rowCount(); i++)
        {
            if (model->item(i)->checkState() == Qt::Checked)
            {
                if (!text.isEmpty())
                {
                    text += ", ";
                }

                text += model->item(i)->text();
            }
        }
        break;
    }

    lineEdit()->setText(text);
}

void CheckList::itemPressed(const QModelIndex &index)
{
    QStandardItem *item = model->itemFromIndex(index);
    item->setCheckState(item->checkState() == Qt::Checked ? Qt::Unchecked : Qt::Checked);
}
