/*
 * This file is part of PokéFinder
 * Copyright (C) 2017-2024 by Admiral_Fish, bumba, and EzPzStreamz
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef SEARCHER_HPP
#define SEARCHER_HPP

#include <Core/Global.hpp>
#include <atomic>
#include <mutex>
#include <vector>

enum class Method : u8;

/**
 * @brief Parent searcher class that stores common attributes
 *
 * @tparam Result Result class that is generated by the searcher
 */
template <class Result>
class SearcherBase
{
public:
    /**
     * @brief Construct a new Searcher object
     *
     * @param method Encounter method
     * @param profile Profile Information
     * @param filter State filter
     */
    SearcherBase() : progress(0), searching(false)
    {
    }

    /**
     * @brief Cancels the running search
     */
    void cancelSearch()
    {
        searching = false;
    }

    /**
     * @brief Returns the progress of the running search
     *
     * @return Progress
     */
    virtual int getProgress() const
    {
        return progress;
    }

    /**
     * @brief Returns the states of the running search
     *
     * @return Vector of computed states
     */
    std::vector<Result> getResults()
    {
        std::lock_guard<std::mutex> guard(mutex);
        auto data = std::move(results);
        return data;
    }

protected:
    std::mutex mutex;
    std::vector<Result> results;
    std::atomic<u32> progress;
    bool searching;
};

/**
 * @brief Parent searcher class that stores common attributes
 *
 * @tparam Profile Profile class that is used by the searcher
 * @tparam Result Result class that is generated by the searcher
 */
template <class Profile, class Result>
class Searcher : public SearcherBase<Result>
{
public:
    Searcher(Method method, const Profile &profile) :
        SearcherBase<Result>(), method(method), profile(profile), tsv(profile.getTID() ^ profile.getSID())
    {
    }

protected:
    Profile profile;
    u16 tsv;
    Method method;
};

#endif // SEARCHER_HPP
