package noria.ui.examples

import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.animateDpAsState
import androidx.compose.animation.core.tween
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.DpOffset
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import fleet.compose.theme.graphics.applyAlpha
import kotlinx.coroutines.delay
import noria.NoriaContext
import noria.state
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import noria.ui.components.scroll
import noria.ui.core.ScrollDirection
import noria.ui.core.ScrollState
import fleet.compose.theme.launchOnce
import fleet.compose.theme.theme
import noria.ui.loop.LocalRenderPerfMetrics
import fleet.compose.theme.components.UiText
import fleet.compose.theme.keys.TextStyleKeys
import noria.ui.withModifier

internal fun rainbowList(): Gallery = gallery("Rainbow List",
                                              NoriaExamples.sourceCodeForFile("RainbowList.kt")) {

  example("Rainbow List") {
    Box(Modifier.heightIn(max = 500.dp).fillMaxSize(), propagateMinConstraints = true) {

      val scrollState = state { ScrollState.empty() }
      val scrollProgressState = animateDpAsState(scrollState.read().position.y, tween(1000, easing = LinearEasing))

      val animatedScrollState = state({
                                        val originalScrollState = scrollState.read()
                                        originalScrollState.copy(
                                          position = DpOffset(originalScrollState.position.x, scrollProgressState.value))
                                      }) { getNewValue ->
        scrollState.update { previousValue -> getNewValue(previousValue) }
      }
      scroll(ScrollDirection.VERTICAL, scrollState = animatedScrollState) {
        Column(Modifier.fillMaxWidth(), propagateMinWidth = true) {
          repeat(10_000) {
            boxWithTintAndBorder(Color.hsv((it * 10f / 360f) % 1f, 0.8f, 1f)) {
              withModifier(Modifier.height(height = 20.dp)) {
              }
              //uiText(it.toString())
            }
          }
        }
      }

      Box(Modifier.matchParentSize(), propagateMinConstraints = true) {
        Box(contentAlignment = Alignment.Center) {
          boxWithTintAndBorder(Color.Black) {
            val renderPerformanceMetrics = LocalRenderPerfMetrics.current
            val fps = renderPerformanceMetrics.averageFpsPerWindow().values.first().toInt()
            UiText(fps.toString(),
                   color = Color.White,
                   userTextSpec = with(theme[TextStyleKeys.Default]) { copy(fontSpec = fontSpec.copy(size = 50.sp)) },
                   modifier = Modifier.padding(24.dp))
          }
        }
      }

      launchOnce {
        while (true) {
          scrollState.update { it.copy(position = DpOffset(it.position.x, it.position.y + 1000.dp)) }
          delay(1000)
        }
      }
    }
  }
}


@Composable
internal fun NoriaContext.boxWithTintAndBorder(color: Color, content: @Composable NoriaContext.() -> Unit) {
  withModifier(
    Modifier
      .background(color.applyAlpha(.7f))
      .border(1.5.dp, Color(0.8f * color.red, 0.8f * color.green, 0.8f * color.blue, 0.9f * color.alpha)),
    content
  )
}
