package noria.ui.examples

//import androidx.compose.ui.unit.Pixel
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.width
import androidx.compose.runtime.Composable
import androidx.compose.runtime.ReadOnlyComposable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.focusTarget
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.times
import fleet.compose.foundation.input.inputHandler
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.modifiers.clickable
import noria.model.Propagate
import noria.ui.components.Row
import noria.ui.components.approximatingLazyColumn
import noria.ui.components.scroll
import noria.ui.core.ScrollDirection
import noria.ui.withModifier
import kotlin.random.Random


const val ROW_MIN_COUNT = 20
const val ROW_MAX_COUNT = 1000


val BASE_HEIGHT = 90.dp
val DISTRIBUTE_HEIGHT = 550.dp


internal typealias ItemIndex = Int

internal typealias ItemHeight = Int

internal typealias ItemsCount = Int

internal typealias HeightFn = (ItemHeight) -> Dp

data class ExampleItem (
    val index : ItemIndex,
    val height : Dp,
    val count : ItemHeight
)

internal typealias RenderFn = (noria.NoriaContext.(ExampleItem) -> Unit)

internal typealias ItemFn = (ItemsCount, HashMap<ItemIndex, Dp>, HeightFn, @Composable RenderFn) -> @Composable (noria.ReadScope.(ItemHeight) -> Row)


internal fun listLazyApproximationExamples(): Gallery {
    return gallery("List View (Lazy Approx.)", NoriaExamples.sourceCodeForFile("ListLazyApproximation.kt")) {

        val randomHeight : HeightFn = { i ->
            BASE_HEIGHT + Random.nextFloat() * DISTRIBUTE_HEIGHT
        }

        val constantHeight : HeightFn = { i ->
            BASE_HEIGHT
        }

        val increasingHeightMin : HeightFn = { i ->
            BASE_HEIGHT + (i * (DISTRIBUTE_HEIGHT / ROW_MIN_COUNT))
        }

        val decreasingHeightMin : HeightFn = { i ->
            BASE_HEIGHT + ((ROW_MIN_COUNT - i) * (DISTRIBUTE_HEIGHT / ROW_MIN_COUNT))
        }

        val increasingHeightMax : HeightFn = { i ->
            BASE_HEIGHT + (i * (DISTRIBUTE_HEIGHT / ROW_MAX_COUNT))
        }

        val decreasingHeightMax : HeightFn = { i ->
            BASE_HEIGHT + ((ROW_MAX_COUNT - i) * (DISTRIBUTE_HEIGHT / ROW_MAX_COUNT))
        }

        val renderText : @Composable RenderFn = { (index, height) ->
            withModifier(modifier = Modifier.height(height)) {
                UiText("$index ($height)")
            }
        }

        val itemFn : @Composable ItemFn = { count, heightMap, heightFn, renderFn ->
            @Composable @ReadOnlyComposable { i ->
                Row(key = i) {
                  val focusRequester = remember { FocusRequester() }
                  withModifier(Modifier
                                 .focusRequester(focusRequester)
                                 .focusTarget()
                                 .inputHandler {
                                   println(i)
                                   Propagate.CONTINUE
                                 }
                                 .clickable(onClick = {
                            println("requestFocus $i")
                                   focusRequester.requestFocus()
                            Propagate.CONTINUE
                        })) {
                            val height = heightMap.computeIfAbsent(i, heightFn)
                            renderText(ExampleItem(i, height, count))
                        }
                }
            }
        }

        example("List with Random Height Elements (10) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MIN_COUNT) }
                    approximatingLazyColumn(count = ROW_MIN_COUNT, nth = itemFn(ROW_MIN_COUNT, heightMap, randomHeight, renderText))
                }
            }
        }

        example("List with Constant Height Elements (10) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MIN_COUNT) }
                    approximatingLazyColumn(count = ROW_MIN_COUNT, nth = itemFn(ROW_MIN_COUNT, heightMap, constantHeight, renderText))
                }
            }
        }

        example("List with Increasing Height Elements (10) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MIN_COUNT) }
                    approximatingLazyColumn(count = ROW_MIN_COUNT, nth = itemFn(ROW_MIN_COUNT, heightMap, increasingHeightMin, renderText))
                }
            }
        }

        example("List with Decreasing Height Elements (10) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<Int, Dp>(ROW_MIN_COUNT) }
                    approximatingLazyColumn(count = ROW_MIN_COUNT, nth = itemFn(ROW_MIN_COUNT, heightMap, decreasingHeightMin, renderText))
                }
            }
        }

        example("List with Random Height Elements (1000) (Lazy Height Approximation)") {
            scroll(direction = ScrollDirection.VERTICAL) {
                val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MAX_COUNT) }
                approximatingLazyColumn(count = ROW_MIN_COUNT, nth = itemFn(ROW_MAX_COUNT, heightMap, randomHeight, renderText))
            }
        }

        example("List with Constant Height Elements (1000) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MAX_COUNT) }
                    approximatingLazyColumn(count = ROW_MAX_COUNT, nth = itemFn(ROW_MAX_COUNT, heightMap, constantHeight, renderText))
                }
            }
        }

        example("List with Increasing Height Elements (1000) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MAX_COUNT) }
                    approximatingLazyColumn(count = ROW_MAX_COUNT, nth = itemFn(ROW_MAX_COUNT, heightMap, increasingHeightMax, renderText))
                }
            }
        }

        example("List with Decreasing Height Elements (1000) (Lazy Height Approximation)") {
            withModifier(Modifier.height(500.dp).width(500.dp)) {
                scroll(direction = ScrollDirection.VERTICAL) {
                    val heightMap = remember { HashMap<ItemIndex, Dp>(ROW_MAX_COUNT) }
                    approximatingLazyColumn(count = ROW_MAX_COUNT, nth = itemFn(ROW_MAX_COUNT, heightMap, decreasingHeightMax, renderText))
                }
            }
        }

    }
}