package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.staticCompositionLocalOf
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import kotlinx.coroutines.delay
import noria.Cell
import noria.NoriaContext
import noria.state
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import noria.ui.components.textInput
import noria.ui.components.textInputModel
import noria.ui.core.boundary
import fleet.compose.theme.launchOnce
import fleet.compose.theme.theme
import noria.ui.loop.LocalRenderPerfMetrics
import fleet.compose.theme.components.UiText
import fleet.compose.theme.keys.TextStyleKeys
import noria.ui.withModifier
import kotlin.math.pow
import kotlin.math.roundToInt

internal fun colorSparks(): Gallery = gallery("Color Sparks",
                                              NoriaExamples.sourceCodeForFile("ColorSparks.kt")) {

  example("Color Sparks") {
    val subdivisionDepthInputModel = textInputModel("7")
    val activeBoxCountInputModel = textInputModel("1")
    val subdivisionDepth = subdivisionDepthInputModel.state.read().content.toIntOrNull() ?: 7
    val totalBoxCount = 4f.pow(subdivisionDepth).roundToInt()
    val reconcilingBoxCount = activeBoxCountInputModel.state.read().content.toIntOrNull()?.coerceAtMost(totalBoxCount) ?: 1

    val boxHueState = state { 0f }
    launchOnce {
      while (true) {
        delay(10)
        boxHueState.update { (it + 0.47f) % 1f }
      }
    }
    Column {
      Row(verticalAlignment = Alignment.CenterVertically) {
        UiText("Subdivision Depth")
        Spacer(Modifier.width(4.dp))
        textInput(subdivisionDepthInputModel)
        Spacer(Modifier.width(32.dp))
        UiText("Active Box Count")
        Spacer(Modifier.width(4.dp))
        textInput(activeBoxCountInputModel, suffixBuilder = { UiText("/ $totalBoxCount") })
      }
      Spacer(Modifier.height(4.dp))
      Row {
        withModifier(Modifier.size(512.dp, 512.dp)) {
    withFpsOverlay {
    CompositionLocalProvider(LocalBoxHueCell provides boxHueState) {
      boundary {
        recursivelySplittingBox(DpSize(512.dp, 512.dp), 0, subdivisionDepth, reconcilingBoxCount)
      }
    }
  }
        }
      }
    }
  }
}

internal val LocalBoxHueCell = staticCompositionLocalOf<Cell<Float>> { error("LocalBoxHueCell is not provided") }

@Composable
private fun NoriaContext.recursivelySplittingBox(size: DpSize, depth: Int, targetDepth: Int, activeBoxCount: Int) {
  boundary {
    if (depth == targetDepth) {
      require(activeBoxCount <= 1) { "wat" }

      val isActive = activeBoxCount == 1
      Spacer(
        Modifier
          .background(if (isActive) randomBrightColor(LocalBoxHueCell.current.read()) else randomGrayTone())
          .size(size.width, size.height)
      )
    }
    else {
      val reconcilingRemainderDistribution = when (activeBoxCount % 4) {
        0 -> listOf(0, 0, 0, 0)
        1 -> listOf(1, 0, 0, 0)
        2 -> listOf(1, 0, 1, 0)
        3 -> listOf(1, 1, 1, 0)
        else -> error("wat")
      }
      Column {
        Row {
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth,
                                  activeBoxCount / 4 + reconcilingRemainderDistribution[0])
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth,
                                  activeBoxCount / 4 + reconcilingRemainderDistribution[1])
        }
        Row {
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth,
                                  activeBoxCount / 4 + reconcilingRemainderDistribution[2])
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth,
                                  activeBoxCount / 4 + reconcilingRemainderDistribution[3])
        }
      }
    }
  }
}

internal fun randomBrightColor(hue: Float): Color = Color.hsv(hue, 0.5f + kotlin.random.Random.nextFloat() * 0.5f,
                                                              0.5f + kotlin.random.Random.nextFloat() * 0.5f)

internal fun randomGrayTone(): Color = Color.hsv(0f, 0f, 0.25f + kotlin.random.Random.nextFloat() * 0.5f)

@Composable
internal fun NoriaContext.withFpsOverlay(content: @Composable NoriaContext.() -> Unit) {
  Box(Modifier.fillMaxSize(), propagateMinConstraints = true) {
    content()

    Box(Modifier.matchParentSize(), propagateMinConstraints = true) {
      boundary {
        Box(contentAlignment = Alignment.Center) {
          boxWithTintAndBorder(Color.Black) {
            val renderPerformanceMetrics = LocalRenderPerfMetrics.current
            val fpsState = state { renderPerformanceMetrics.averageFpsPerWindow().values.first().roundToInt() }
            UiText(fpsState.read().toString(),
                   color = Color.White,
                   userTextSpec = with(theme[TextStyleKeys.Default]) { copy(fontSpec = fontSpec.copy(size = 50.sp)) },
                   modifier = Modifier.padding(24.dp))

            launchOnce {
              while (true) {
                fpsState.update { renderPerformanceMetrics.averageFpsPerWindow().values.first().roundToInt() }
                delay(500)
              }
            }
          }
        }
      }
    }
  }
}
