package noria.ui.examples

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.focusTarget
import androidx.compose.ui.unit.dp
import fleet.compose.foundation.input.inputHandler
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.itemCellColors
import fleet.compose.theme.components.modifiers.clickable
import noria.model.Propagate
import noria.state
import noria.ui.components.*
import noria.ui.components.list.*
import noria.ui.core.ScrollDirection
import noria.ui.withModifier


internal fun listViewGallery(): Gallery {
  return gallery("List View", NoriaExamples.sourceCodeForFile("ListView.kt")) {

    example("List") {
      withModifier(Modifier.height(height = 300.dp)) {
        listView(listModel(
          size = 10000,
          onConfirm = { items -> println("confirmed items: $items"); Propagate.STOP }
        ) { idx ->
          ListItem(idx, idx) { opts ->
            defaultListCell(text = idx.toString(),
                            cellColors = { itemCellColors(it) },
                            listItemOpts = opts)
          }
        }, stretchHorizontally = true)
      }
    }

    example("List with Multiple Selection") {
      withModifier(Modifier.height(height = 300.dp)) {
        listView(listModel(size = 10000,
                           options = ListViewOptions(multiSelectionEnabled = true),
                           onConfirm = { items -> println("confirmed items: $items"); Propagate.STOP }
        ) { idx ->
          ListItem(idx, idx) { opts ->
            defaultListCell(text = idx.toString(),
                            cellColors = { itemCellColors(it) },
                            listItemOpts = opts)
          }
        }, stretchHorizontally = true)
      }
    }

    example("List with SpeedSearch") {
      withModifier(Modifier.height(height = 300.dp)) {
        listView(listModel(10000, options = ListViewOptions(speedSearchOptions = SpeedSearchOptions.Default()),
                           onConfirm = { items -> println("confirmed items: $items"); Propagate.STOP }) { idx ->
          ListItem(idx,
                   idx,
                   textToMatch = idx.toString()) { opts ->
            defaultListCell(text = idx.toString(),
                            cellColors = { itemCellColors(it) },
                            listItemOpts = opts)
          }
        }, stretchHorizontally = true)
      }
    }

    example("List with Custom SpeedSearch") {
      withModifier(Modifier.height(height = 300.dp)) {
        Column {
          val state = state { createTextInputData() }
          val hasResults = state { true }
          val textInputStyle = if (hasResults.read()) {
            defaultTextInputStyle()
          }
          else {
            errorTextInputStyle()
          }

          textInput(textInputModel(state), style = textInputStyle)
          listView(listModel(10000, options = ListViewOptions(speedSearchOptions = SpeedSearchOptions.Custom(state { true }, state, hasResults)),
                             onConfirm = { items -> println("confirmed items: $items"); Propagate.STOP }) { idx ->
            ListItem(idx,
                     idx,
                     textToMatch = idx.toString()) { opts ->
              defaultListCell(text = idx.toString(),
                              cellColors = { itemCellColors(it) },
                              listItemOpts = opts)
            }
          }, stretchHorizontally = true)
        }
      }
    }

    example("Height Key Based Lazy Column Algorithm") {
      withModifier(Modifier.size(width = 500.dp, height = 500.dp)) {
        scroll(direction = ScrollDirection.VERTICAL) {
          heightKeyBasedLazyColumn(size = 10000) { i ->
            Row(key = i) {
              val focusRequester = remember { FocusRequester() }
                UiText(
                  "$i",
                  modifier = Modifier
                    .focusRequester(focusRequester)
                    .focusTarget()
                    .inputHandler {
                      println(i)
                      Propagate.CONTINUE
                    }
                    .clickable(onClick = {
                      println("requestFocus $i")
                      focusRequester.requestFocus()
                    Propagate.CONTINUE
                  })
                )
            }
          }
        }
      }
    }
  }
}