package noria.ui.examples

import androidx.compose.foundation.*
import androidx.compose.foundation.gestures.*
import androidx.compose.foundation.interaction.*
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.selection.toggleable
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Paint
import androidx.compose.ui.graphics.drawscope.ContentDrawScope
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.input.pointer.*
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.round
import androidx.compose.ui.util.fastAny
import androidx.compose.ui.util.fastFirstOrNull
import fleet.compose.foundation.culling.layout.CullingLayout
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.graphics.applyAlpha
import noria.*
import noria.ui.components.*
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import noria.ui.core.*
import fleet.compose.theme.components.UiText
import fleet.compose.theme.overlay.overlay
import kotlin.random.Random

@OptIn(ExperimentalFoundationApi::class, ExperimentalComposeUiApi::class)
internal fun composePointerInputModifiersExamples(): Gallery = gallery("Compose Pointer Input Modifiers",
                                                                       NoriaExamples.sourceCodeForFile("ComposePointerInputModifiers.kt")) {
  example("Toggleable") {
    Column {
      var enabled by remember { mutableStateOf(true) }
      Checkbox(enabled, label = "Toggling Enabled", onCheckedChange = {
        enabled = !it
      })
      var toggled by remember { mutableStateOf(false) }
      UiText(
        if (toggled) "On" else "Off",
        modifier = Modifier
          .toggleable(toggled, enabled = enabled) { toggled = it }
          .background(Color.Red)
          .size(100.dp)
      )
    }
  }

  example("Pointer Icon") {
    Column {
      var pointerIcon by remember { mutableStateOf(PointerIcon.Crosshair) }
      dropDownList(
        listOf(PointerIcon.Default, PointerIcon.Crosshair, PointerIcon.Hand, PointerIcon.Text),
        pointerIcon, { it.toString() }) {
        pointerIcon = it
      }
      UiText(
        "Hover Me",
        modifier = Modifier
          .pointerHoverIcon(pointerIcon)
          .background(Color.Red)
          .size(100.dp)
      )
    }
  }

  example("Chained Modifiers on a Single Node") {
    Column {
      var shouldConsume by remember { mutableStateOf(true) }
      var outerBoxColor by remember { mutableStateOf(Color.Blue) }
      var innerBoxColor by remember { mutableStateOf(Color.Green) }
      Checkbox(shouldConsume, label = "Consume", onCheckedChange = {
        shouldConsume = !it
      })
      Spacer(Modifier.height(5.dp))
      Box(
        Modifier
          .background(outerBoxColor)
          .onPointerEvent(PointerEventType.Press) { pressEvent ->
            if (pressEvent.changes.fastAny { it.changedToDown() }) {
              outerBoxColor = Color.random()
            }
          }
          .padding(all = 30.dp)
          .background(innerBoxColor)
          .onPointerEvent(PointerEventType.Press) { pressEvent ->
            pressEvent.changes.fastFirstOrNull<PointerInputChange> { it.changedToDown() }?.let { press ->
              if (shouldConsume) {
                press.consume()
              }
              innerBoxColor = Color.random()
            }
          }
          .size(100.dp)
      )
    }
  }

  /**
   * Providing any of [onDoubleClick] or [onLongClick] will introduce a noticeable delay before the simple [onClick] handler is run!
   */
  example("Click Combined with Long Click and/or Double Click") {
    Column {
      var eventText by remember { mutableStateOf("no event" to 0) }
      UiText(
        "${eventText.first} – ${eventText.second}",
        modifier = Modifier
          .onClick(
            onDoubleClick = { eventText = "on double click" to Random.nextInt(100) },
            onLongClick = { eventText = "on long click" to Random.nextInt(100) },
            onClick = { eventText = "on click" to Random.nextInt(100) }
          )
          .background(Color.Red)
          .size(200.dp, 100.dp),
      )
    }
  }

  example("Nested Clickables") {
    Column {
      var backgroundColor by remember { mutableStateOf(Color.Green) }
      var foregroundColor by remember { mutableStateOf(Color.Red) }
      Box(Modifier
            .background(backgroundColor)
            .clickable {
              println("Outer click")
              backgroundColor = Color.random()
            }
            .padding(all = 30.dp)) {
        Box(Modifier
              .clickable {
                println("Inner click")
                foregroundColor = Color.random()
              }
              .background(foregroundColor)
              .size(100.dp))
      }
    }
  }

  example("Click") {
    Column {
      var color by remember { mutableStateOf(Color.Black) }
      Box(Modifier
            .onClick(onClick = { color = Color.random() })
            .background(color)
            .size(100.dp))
    }
  }

  example("Pressed State via Interaction Source") {
    Column {
      val interactionSource = remember { MutableInteractionSource() }
      val pressed by interactionSource.noriaCollectIsPressedAsState(this)
      val backgroundColor = if (pressed) Color.Red else Color.Blue
      Box(Modifier
            .onClick(
              onClick = { println("clicked") },
              interactionSource = interactionSource
            )
            .background(backgroundColor)
            .size(100.dp)) {
        UiText(if (pressed) "pressed" else "not pressed")
      }
    }
  }

  example("Hover State via Interaction Source") {
    Column {
      val interactionSource = remember { MutableInteractionSource() }
      val hovered by interactionSource.noriaCollectIsHoveredAsState(this)
      val backgroundColor = if (hovered) Color.Cyan else Color.Gray
      UiText(
        if (hovered) "hovered" else "not hovered",
        modifier = Modifier
          .hoverable(interactionSource)
          .background(backgroundColor)
          .size(100.dp)
      )
    }
  }

  example("Pressed and Hovered States via Interaction Source and Indication") {
    Column {
      val interactionSource = remember { MutableInteractionSource() }
      UiText(
        "button text",
        modifier = Modifier
          .clickable(interactionSource, indication = DebugIndication) { println("clicked") }
          .size(100.dp, 150.dp)
      )
    }
  }

  example("Tap") {
    Column {
      var offset by remember { mutableStateOf(null as Offset?) }
      Box(
        Modifier
          .size(500.dp)
          .background(Color.LightGray)
          .pointerInput(Unit) {
            detectTapGestures {
              offset = it
            }
          },
        contentAlignment = Alignment.Center
      ) {
        UiText(offset?.toString() ?: "no tapped offset")
      }
    }
  }

  example("Draggable") {
    Column {
      var offset by remember { mutableStateOf(Offset.Zero) }
      // TODO[dragging]: noria does not render views outside of viewport. see `noria.ui.core.childViewport`
      Box(Modifier
            .size(500.dp)
            .background(Color.LightGray)
            .pointerInput(Unit) {
              detectDragGestures { change, dragAmount ->
                change.consume()
                offset += dragAmount
              }
            }
      ) {
        Box(
          Modifier
            .background(Color.Cyan)
            .offset { offset.round() }
            .background(Color.Blue)
            .size(50.dp)
            .padding(10.dp)
        ) {
          CullingLayout { constraints ->
            layout(constraints.maxWidth, constraints.maxHeight) { visibleBounds ->
              Box(Modifier.fillMaxSize().background(Color.Yellow)) {
                Box(
                  Modifier
                    .absolutePhysicalPadding(visibleBounds.left, visibleBounds.top)
                    .size(visibleBounds.width.toDp(), visibleBounds.height.toDp())
                    .background(Color.Magenta.applyAlpha(0.5f))
                )
              }
            }
          }
        }
      }
    }
  }

  example("Event Propagation") {
    Column {
      Row(Modifier.fillMaxSize()) {
        Box(
          Modifier
            .logPointerInput("Green")
            .background(Color.Green)
            .size(50.dp)
        )
        Box(
          Modifier
            .logPointerInput("Blue")
            .background(Color.Blue)
            .padding(30.dp),
          propagateMinConstraints = true
        ) {
          Box(
            Modifier
              .logPointerInput("Red")
              .background(Color.Red)
              .padding(30.dp),
            propagateMinConstraints = true
          ) {
            Box(
              Modifier
                .background(Color.Yellow)
                .logPointerInput("Yellow")
                .size(30.dp)
            )
          }
        }
        Box(
          Modifier
            .logPointerInput("Magenta", inboundDragEnabled = true)
            .background(Color.Magenta)
            .padding(30.dp)
        ) {
          UiText("inbound drag enabled")
        }
        Box(
          Modifier
            .logPointerInput("Cyan")
            .background(Color.Cyan.applyAlpha(0.5f))
            .size(50.dp)
        )
        Box(
          Modifier
            .offset((-75).dp, 25.dp)
            .logPointerInput("Red")
            .background(Color.Red.applyAlpha(0.5f))
            .padding(30.dp)
        ) {
          UiText("sharing with siblings enabled")
        }
      }
    }
  }

  example("Draggable with Noria Overlay") {
    Column {
      var offset by remember { mutableStateOf(Offset.Zero) }
      Box(modifier = Modifier.overlay(MainOverlayHost) {
        Spacer(
          Modifier
            .align(Alignment { overlaySize, hostSize, _ ->
              anchorBounds.topLeft + offset.round()
            })
            .background(Color.Red)
            .size(50.dp)
        )
      }) {
        Box(modifier = Modifier
          .size(500.dp)
          .background(Color.LightGray)
          .pointerInput(Unit) {
            detectDragGestures { change, dragAmount ->
              change.consume()
              offset += dragAmount
            }
          }
        )
      }
    }
  }

  example("Draggable Noria Overlay") {
    Column {
      var offset by remember { mutableStateOf(Offset.Zero) }
      Box(modifier = Modifier.overlay(MainOverlayHost) {
        Spacer(
          Modifier
            .background(Color.Green)
            .offset { offset.round() }
            .alignInAnchor(Alignment.TopStart)
            .pointerInput(Unit) {
              detectDragGestures { change, dragAmount ->
                change.consume()
                offset += dragAmount
              }
            }
            .background(Color.Red)
            .size(50.dp)
        )
      }) {
        Box(modifier = Modifier
          .size(500.dp)
          .background(Color.LightGray)
        )
      }
    }
  }

  example("Enter on Appearance Below Cursor") {
    Column {
      var offset by remember { mutableStateOf<Offset?>(null) }
      Box(modifier = Modifier
        .size(500.dp)
        .background(Color.LightGray)
        .pointerHoverIcon(PointerIcon.Crosshair)
        .pointerInput(Unit) {
          awaitPointerEventScope {
            while (true) {
              val pointerEvent = awaitPointerEvent()
              if (offset != null) {
                println("Outer: ${pointerEvent.type}")
              }
            }
          }
        }
        .onPointerEvent(PointerEventType.Press) {
          offset = it.changes[0].position - Offset(6f, 6f)
        }) {
        offset?.let { nonNullOffset ->
          Box(Modifier
                .offset { nonNullOffset.round() }
                .background(Color.Red)
                .size(50.dp)
                .mouseBlocker())
        }
      }
    }
  }
}

private object DebugIndication : Indication {
  class DefaultIndicationInstance(
    val isPressed: State<Boolean>,
    val isHovered: State<Boolean>
  ) : IndicationInstance {
    override fun ContentDrawScope.drawIndication() {
      drawContent()
      drawIntoCanvas { canvas ->
        if (isPressed.value) {
          val paint = Paint().apply {
            color = Color.Red.applyAlpha(0.5f)
          }
          canvas.drawRect(Rect(Offset.Zero, size.copy(height = size.height / 2)), paint)
        }

        if (isHovered.value) {
          val paint = Paint().apply { color = Color.Green.applyAlpha(0.5f) }
          canvas.drawRect(Rect(Offset(0f, size.height / 2), size.copy(height = size.height / 2)), paint)
        }
      }
    }
  }

  @Composable
  override fun NoriaContext.rememberUpdatedInstance(interactionSource: InteractionSource): IndicationInstance {
    val isHovered = interactionSource.noriaCollectIsHoveredAsState(this)
    val isPressed = interactionSource.noriaCollectIsPressedAsState(this)
    return DefaultIndicationInstance(isPressed, isHovered)
  }
}


private val rnd = Random.Default
internal fun Color.Companion.random(): Color = Color(
  red = rnd.nextInt(256),
  green = rnd.nextInt(256),
  blue = rnd.nextInt(256)
)

private fun Modifier.logPointerInput(label: String,
                                     shareWithNodesBehind: Boolean = true,
                                     interceptOutOfBoundsDragEventsAfterPress: Boolean = true,
                                     inboundDragEnabled: Boolean = false): Modifier = pointerInput(Unit) {
  this.shareWithNodesBehind = shareWithNodesBehind
  this.interceptOutOfBoundsDragEventsAfterPress = interceptOutOfBoundsDragEventsAfterPress
  this.inboundDragEnabled = inboundDragEnabled
  awaitPointerEventScope {
    while (true) {
      val pointerEvent = awaitPointerEvent()
      println("$label: ${pointerEvent.type}")
    }
  }
}
