package noria.ui.examples

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.layout.Arrangement.Absolute.spacedBy
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import fleet.compose.theme.ai.AiSplitButtonStyle
import fleet.compose.theme.components.*
import fleet.compose.theme.keys.ThemeKeys
import noria.NoriaContext
import noria.model.IconKey
import noria.model.ThemeKey
import noria.scope
import fleet.compose.theme.components.MenuItem
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import noria.ui.components.scroll
import noria.ui.core.ScrollDirection
import noria.ui.events.HandlerScope
import noria.ui.events.Input
import noria.ui.text.Link
import fleet.compose.theme.components.UiText
import noria.ui.text.DropdownLink
import noria.windowManagement.api.ModifiersState
import noria.windowManagement.extensions.COMMAND_WINDOWS_SUPER_ONLY
import noria.windowManagement.extensions.Keys

private val hintTextOptions = listOf(null, Input.Keystroke(Keys.RightBracket, ModifiersState.COMMAND_WINDOWS_SUPER_ONLY).presentableName)
private val iconOptions = listOf(null, ThemeKeys.RightIcon)

internal fun buttonExamples(): Gallery = gallery("Buttons", NoriaExamples.sourceCodeForFile("Buttons.kt")) {
  example("Primary") {
    @Suppress("COMPOSABLE_FUNCTION_REFERENCE")
    placeButtonCombinations(
      { text, iconKey, iconSide, hintText, enabled, onClick -> LargePrimaryButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> PrimaryButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> SmallPrimaryButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) }
    )
  }

  example("Secondary") {
    @Suppress("COMPOSABLE_FUNCTION_REFERENCE")
    placeButtonCombinations(
      { text, iconKey, iconSide, hintText, enabled, onClick -> LargeButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> Button(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> SmallButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
    )
  }

  example("Dangerous") {
    @Suppress("COMPOSABLE_FUNCTION_REFERENCE")
    placeButtonCombinations(
      { text, iconKey, iconSide, hintText, enabled, onClick -> LargeDangerousButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> DangerousButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> SmallDangerousButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
    )
  }

  example("Positive") {
    @Suppress("COMPOSABLE_FUNCTION_REFERENCE")
    placeButtonCombinations(
      { text, iconKey, iconSide, hintText, enabled, onClick -> LargeAcceptButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> AcceptButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> SmallAcceptButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
    )
  }

  example("Warning") {
    @Suppress("COMPOSABLE_FUNCTION_REFERENCE")
    placeButtonCombinations(
      { text, iconKey, iconSide, hintText, enabled, onClick -> LargeWarningButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> WarningButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
      { text, iconKey, iconSide, hintText, enabled, onClick -> SmallWarningButton(Modifier, text, iconKey, iconSide, hintText, enabled, onClick) },
    )
  }

  example("Toggle Button") {
    Row(verticalAlignment = Alignment.CenterVertically) {
      for (icon in iconOptions) {
        for (hintText in hintTextOptions) {
          scope(icon to hintText) {
            Column(horizontalAlignment = Alignment.CenterHorizontally, modifier = Modifier.padding(4.dp)) {
              val selectedState = remember { mutableStateOf(false) }
              ToggleButton(selected = selectedState.value, text = "Text", iconKey = icon, hintText = hintText, onClick = {
                selectedState.value = !selectedState.value
              })
              Spacer(Modifier.height(4.dp))
              ToggleButton(selected = false, text = "Text", iconKey = icon, hintText = hintText, enabled = false, onClick = {})
              Spacer(Modifier.height(4.dp))
              ToggleButton(selected = true, text = "Text", iconKey = icon, hintText = hintText, enabled = false, onClick = {})
            }
          }
        }
      }
    }
  }

  example("Split Button") {
    @Composable
    fun NoriaContext.SplitButtonExample(style: ButtonStyle, text: String = "Action 1") {
      Row(horizontalArrangement = Arrangement.spacedBy(8.dp)) {
        SplitButton(
          text = text,
          style = style,
          menuActions = listOf(
            MenuItem.Action("Action 2") {},
            MenuItem.Action("Action 3") {},
            MenuItem.Action("Action 4") {},
          )
        ) {}
        SplitButton(
          text = text,
          style = style,
          enabled = false,
          menuActions = listOf(
            MenuItem.Header("Header"),
            MenuItem.Separator,
            MenuItem.Action("Action 2") {},
            MenuItem.Action("Action 3") {},
            MenuItem.Action("Action 4") {},
          )
        ) {}
      }

    }

    Column(verticalArrangement = Arrangement.spacedBy(8.dp)) {
      SplitButtonExample(PrimarySplitButtonStyle())
      SplitButtonExample(SecondarySplitButtonStyle())
      SplitButtonExample(DangerousSplitButtonStyle())
      SplitButtonExample(WarningSplitButtonStyle())
      SplitButtonExample(PositiveSplitButtonStyle())
      SplitButtonExample(AiSplitButtonStyle())
      SplitButtonExample(PrimarySplitButtonStyle(), " Long long long button text")
    }
  }

  example("Menu Button") {
    Column {
      MenuButton(
        text = "Action 1",
        menuActions = listOf(
          MenuItem.Header("Header"),
          MenuItem.Separator,
          MenuItem.Action("Action 2") {},
          MenuItem.Action("Action 3") {},
          MenuItem.Action("Action 4") {},
        )
      )
      Spacer(Modifier.height(4.dp))
      MenuButton(
        text = "Action 1",
        menuActions = listOf(),
        enabled = false
      )
    }
  }
  val icon = IconKey("icons.settings")

  example("Ghost Button") {
    Column(verticalArrangement = spacedBy(6.dp)) {
      Row(horizontalArrangement = spacedBy(4.dp), verticalAlignment = Alignment.CenterVertically) {
        UiText(modifier = Modifier.width(54.dp), text = "Default")
        GhostButton(iconKey = icon) { }
        GhostButton(iconKey = icon, enabled = false) { }
        Spacer(Modifier.width(16.dp))
        GhostButton(iconKey = icon, text = "Label") { }
        GhostButton(iconKey = icon, text = "Label", enabled = false) { }
        Spacer(Modifier.width(16.dp))
        GhostButton(text = "Label") { }
        GhostButton(text = "Label", enabled = false) { }
      }
      Row(horizontalArrangement = spacedBy(4.dp), verticalAlignment = Alignment.CenterVertically) {
        UiText(modifier = Modifier.width(54.dp), text = "Large")
        LargeGhostButton(iconKey = icon) { }
        LargeGhostButton(iconKey = icon, enabled = false) { }
        Spacer(Modifier.width(16.dp))
        LargeGhostButton(iconKey = icon, text = "Label") { }
        LargeGhostButton(iconKey = icon, text = "Label", enabled = false) { }
        Spacer(Modifier.width(16.dp))
        LargeGhostButton(text = "Label") { }
        LargeGhostButton(text = "Label", enabled = false) { }
      }
      Row(horizontalArrangement = spacedBy(4.dp), verticalAlignment = Alignment.CenterVertically) {
        UiText(modifier = Modifier.width(54.dp), text = "Small")
        SmallGhostButton(iconKey = icon) { }
        SmallGhostButton(iconKey = icon, enabled = false) { }
        Spacer(Modifier.width(16.dp))
        SmallGhostButton(iconKey = icon, text = "Label") { }
        SmallGhostButton(iconKey = icon, text = "Label", enabled = false) { }
        Spacer(Modifier.width(16.dp))
        SmallGhostButton(text = "Label") { }
        SmallGhostButton(text = "Label", enabled = false) { }
      }
      Row(horizontalArrangement = spacedBy(4.dp), verticalAlignment = Alignment.CenterVertically) {
        UiText(modifier = Modifier.width(54.dp), text = "Tiny")
        TinyGhostButton(iconKey = icon) { }
        TinyGhostButton(iconKey = icon, enabled = false) { }
        Spacer(Modifier.width(16.dp))
        TinyGhostButton(iconKey = icon, text = "Label") { }
        TinyGhostButton(iconKey = icon, text = "Label", enabled = false) { }
        Spacer(Modifier.width(16.dp))
        TinyGhostButton(text = "Label") { }
        TinyGhostButton(text = "Label", enabled = false) { }
      }
    }
  }

  example("Ghost Toggle Button") {
    Column(verticalArrangement = spacedBy(6.dp)) {
      Row(horizontalArrangement = spacedBy(4.dp), verticalAlignment = Alignment.CenterVertically) {
        UiText(modifier = Modifier.width(54.dp), text = "Default")

        val toggleState = remember { mutableStateOf(true) }
        GhostToggleButton(iconKey = icon, selected = toggleState.value, onClick = { toggleState.value = !toggleState.value })
        val toggleState2 = remember { mutableStateOf(false) }
        GhostToggleButton(iconKey = icon, selected = toggleState2.value, onClick = { toggleState2.value = !toggleState2.value })
        GhostToggleButton(enabled = false, iconKey = icon, selected = true, onClick = { })
        GhostToggleButton(enabled = false, iconKey = icon, selected = false, onClick = { })
        Spacer(Modifier.width(16.dp))

        val toggleState3 = remember { mutableStateOf(true) }
        GhostToggleButton(iconKey = icon, text = "Label", selected = toggleState3.value, onClick = { toggleState3.value = !toggleState3.value })
        val toggleState4 = remember { mutableStateOf(false) }
        GhostToggleButton(iconKey = icon, text = "Label", selected = toggleState4.value, onClick = { toggleState4.value = !toggleState4.value })
        GhostToggleButton(enabled = false, text = "Label", iconKey = icon, selected = true, onClick = { })
        GhostToggleButton(enabled = false, text = "Label", iconKey = icon, selected = false, onClick = { })
        Spacer(Modifier.width(16.dp))

        val toggleState5 = remember { mutableStateOf(true) }
        GhostToggleButton(text = "Label", selected = toggleState5.value, onClick = { toggleState5.value = !toggleState5.value })
        val toggleState6 = remember { mutableStateOf(false) }
        GhostToggleButton(text = "Label", selected = toggleState6.value, onClick = { toggleState6.value = !toggleState6.value })
        GhostToggleButton(enabled = false, text = "Label", selected = true, onClick = { })
        GhostToggleButton(enabled = false, text = "Label", selected = false, onClick = { })
      }
      Row(horizontalArrangement = spacedBy(4.dp), verticalAlignment = Alignment.CenterVertically) {
        UiText(modifier = Modifier.width(54.dp), text = "Large")

        val toggleState = remember { mutableStateOf(true) }
        LargeGhostToggleButton(iconKey = icon, selected = toggleState.value, onClick = { toggleState.value = !toggleState.value })
        val toggleState2 = remember { mutableStateOf(false) }
        LargeGhostToggleButton(iconKey = icon, selected = toggleState2.value, onClick = { toggleState2.value = !toggleState2.value })
        LargeGhostToggleButton(enabled = false, iconKey = icon, selected = true, onClick = { })
        LargeGhostToggleButton(enabled = false, iconKey = icon, selected = false, onClick = { })
        Spacer(Modifier.width(16.dp))

        val toggleState3 = remember { mutableStateOf(true) }
        LargeGhostToggleButton(iconKey = icon, text = "Label", selected = toggleState3.value, onClick = { toggleState3.value = !toggleState3.value })
        val toggleState4 = remember { mutableStateOf(false) }
        LargeGhostToggleButton(iconKey = icon, text = "Label", selected = toggleState4.value, onClick = { toggleState4.value = !toggleState4.value })
        LargeGhostToggleButton(enabled = false, text = "Label", iconKey = icon, selected = true, onClick = { })
        LargeGhostToggleButton(enabled = false, text = "Label", iconKey = icon, selected = false, onClick = { })
        Spacer(Modifier.width(16.dp))

        val toggleState5 = remember { mutableStateOf(true) }
        LargeGhostToggleButton(text = "Label", selected = toggleState5.value, onClick = { toggleState5.value = !toggleState5.value })
        val toggleState6 = remember { mutableStateOf(false) }
        LargeGhostToggleButton(text = "Label", selected = toggleState6.value, onClick = { toggleState6.value = !toggleState6.value })
        LargeGhostToggleButton(enabled = false, text = "Label", selected = true, onClick = { })
        LargeGhostToggleButton(enabled = false, text = "Label", selected = false, onClick = { })
      }
    }
  }
}

@Composable
private fun NoriaContext.placeButtonCombinations(
  vararg builders: @Composable NoriaContext.(
    String,
    ThemeKey<String>?,
    IconSide,
    String?,
    Boolean,
    HandlerScope.() -> Unit,
  ) -> Unit,
) {
  scroll(direction = ScrollDirection.HORIZONTAL) {
    Row(verticalAlignment = Alignment.CenterVertically) {
      for (builder in builders) {
        for (icon in iconOptions) {
          for (hintText in hintTextOptions) {
            scope(
              Triple<@Composable (NoriaContext.(String, ThemeKey<String>?, IconSide, String?, Boolean, HandlerScope.() -> Unit) -> Unit), IconKey?, String?>(
                builder, icon, hintText)) {
              Column(horizontalAlignment = Alignment.CenterHorizontally, modifier = Modifier.padding(4.dp)) {
                builder("Text", icon, IconSide.Left, hintText, true) {}
                Spacer(Modifier.height(4.dp))
                builder("Text", icon, IconSide.Left, hintText, false) {}
              }
            }
          }
        }
      }
    }
  }
}