package noria.ui.examples

import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.FontHinting
import androidx.compose.ui.text.FontRasterizationSettings
import androidx.compose.ui.text.FontSmoothing
import androidx.compose.ui.text.font.FontVariation
import androidx.compose.ui.text.style.BaselineShift
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.unit.Dp
import fleet.compose.foundation.culling.layout.CullingLayout
import fleet.compose.theme.LocalFontRasterizationSettings
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.buildThemedAnnotatedString
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.graphics.applyAlpha
import fleet.compose.theme.keys.TextStyleKeys
import fleet.compose.theme.keys.ThemeKeys
import fleet.compose.theme.text.*
import fleet.compose.theme.theme
import fleet.util.letIfNotNull
import noria.memo
import noria.model.ThemeKey
import noria.scope
import noria.state
import noria.ui.components.dropDownList
import noria.ui.components.innerTextInputStyle
import noria.ui.components.split.SplitDirection
import noria.ui.components.split.splitView
import noria.ui.components.textInput
import noria.ui.components.textInputModel

internal fun textExamples(): Gallery = gallery("Text", NoriaExamples.sourceCodeForFile("Text.kt")) {
  example("Regular UI Text") {
    UiText("Text")
  }

  example("Text Rendering Playground") {
    val labelWidth = 150.dp

    @Composable
    fun <T: Any?> ColumnScope.ParameterInput(label: String, initialValue: T): String {
      val textInputModel = scope(initialValue ?: Unit) {
        textInputModel(initialValue?.toString() ?: "")
      }
      Row {
        UiText(label, modifier = Modifier.width(labelWidth))
        textInput(textInputModel)
      }
      return textInputModel.state.read().content
    }


    @Composable
    fun <T> ColumnScope.ParameterDropDown(label: String, items: List<T>, initialValue: T, presentationFn: (T) -> String): T {
      var state: T by remember<MutableState<T>> { mutableStateOf<T>(initialValue) }
      Row {
        UiText(label, modifier = Modifier.width(labelWidth))
        dropDownList(items, state, presentationFn) { selection ->
          state = selection
        }
      }
      return state
    }

    @Composable
    fun ColumnScope.ParameterCheckbox(label: String, initialValue: Boolean): Boolean {
      var state: Boolean by remember { mutableStateOf(initialValue) }
      Row {
        UiText(label, modifier = Modifier.width(labelWidth))
        Checkbox(state, { checked -> state = checked })
      }
      return state
    }

    Row {
      Column(
        Modifier.padding(4.dp, 8.dp),
        verticalArrangement = Arrangement.spacedBy(4.dp),
      ) {
        val sample = ParameterInput("Sample", "The quick brown fox jumps over the lazy dog")
        val textStyleKey = ParameterDropDown<ThemeKey<TextSpec>>("Base Text Style", TextStyleKeys.allKeys.toList(), TextStyleKeys.Default) {
          it.name
        }
        val textSpec = textSpec(textStyleKey)
        val fontFamily = ParameterInput("Font Family", textSpec.fontSpec.family)
        val fontSize = ParameterInput("Font Size", textSpec.fontSpec.size.value).toFloatOrNull()?.sp ?: 13.sp
        val weight = ParameterInput("Weight", textSpec.fontSpec.weight).toIntOrNull()?.coerceIn(1, 1000) ?: 400
        val initialOpticalSize = textSpec.fontSpec.fontVariations?.settings?.find {
          it.axisName == "opsz"
        }?.toVariationValue(null) ?: 14f
        val opticalSize = ParameterInput("Optical Size", initialOpticalSize).toFloatOrNull() ?: initialOpticalSize
        val italicState = state { false }
        Row {
          UiText("Italic", modifier = Modifier.width(labelWidth))
          Checkbox(italicState.read(), { checked -> italicState.update { checked } })
        }
        val lineHeight = ParameterInput("Line Height", textSpec.lineHeight).toFloatOrNull()
        val letterSpacing = ParameterInput("Letter Spacing", textSpec.letterSpacing.value).toFloatOrNull()?.sp ?: textSpec.letterSpacing
        val wordSpacing = ParameterInput("Word Spacing", textSpec.wordSpacing.value).toFloatOrNull()?.sp ?: textSpec.wordSpacing
        val baselineShift = ParameterInput("Baseline Shift", textSpec.baselineShift.multiplier).toFloatOrNull() ?: 0f
        val platformFontRasterizationSettings = LocalFontRasterizationSettings.current
        val hinting = ParameterDropDown("Hinting", FontHinting.entries.toList(), platformFontRasterizationSettings.hinting) { it.name }
        val smoothing = ParameterDropDown("Smoothing", FontSmoothing.entries.toList(), platformFontRasterizationSettings.smoothing) { it.name }
        val autoHintingForced = ParameterCheckbox("Auto-Hinting Forced", platformFontRasterizationSettings.autoHintingForced)
        val subpixelPositioning = ParameterCheckbox("Subpixel Positioning", platformFontRasterizationSettings.subpixelPositioning)
        val fontRasterizationSettings = FontRasterizationSettings(
          smoothing = smoothing,
          hinting = hinting,
          subpixelPositioning = subpixelPositioning,
          autoHintingForced = autoHintingForced,
        )
        CompositionLocalProvider(LocalFontRasterizationSettings provides fontRasterizationSettings) {
          Box(Modifier.border(width = Dp.Hairline, color = Color.Black)) {
            UiText(
              sample,
              userTextSpec = textSpec.copy(
                fontSpec = FontSpec(
                  family = fontFamily,
                  size = fontSize,
                  weight = weight,
                  italic = italicState.read(),
                  fontVariations = FontVariation.Settings(FontVariation.Setting("opsz", opticalSize)),
                ),
                lineHeight = lineHeight,
                letterSpacing = letterSpacing,
                wordSpacing = wordSpacing,
                baselineShift = BaselineShift(baselineShift),
              ),
            )
          }
        }
      }
    }
  }

  example("Fleet UI Kit Typography") {

    val fonts = listOf(
      "Header0Semibold" to TextStyleKeys.Header0SemiBold,
      "Header1Semibold" to TextStyleKeys.Header1SemiBold,
      "Header1" to TextStyleKeys.Header1,
      "Header2Semibold" to TextStyleKeys.Header2SemiBold,
      "Header2" to TextStyleKeys.Header2,
      "Header3Semibold" to TextStyleKeys.Header3SemiBold,
      "Header3" to TextStyleKeys.Header3,
      "Header4Semibold" to TextStyleKeys.Header4SemiBold,
      "Header5Semibold" to TextStyleKeys.Header5SemiBold,
      "Default" to TextStyleKeys.Default,
      "DefaultItalic" to TextStyleKeys.DefaultItalic,
      "DefaultSemiBold" to TextStyleKeys.DefaultSemiBold,
      "DefaultMultiline" to TextStyleKeys.DefaultMultiline,
      "Medium" to TextStyleKeys.Medium,
      "MediumSemibold" to TextStyleKeys.MediumSemiBold,
      "Small" to TextStyleKeys.Small,
      "Code" to TextStyleKeys.Code,
      "CodeItalic" to TextStyleKeys.CodeItalic,
      "CodeBold" to TextStyleKeys.CodeBold,
    )

    Row {
      Column {
        fonts.forEach { (text, key) ->
          Row(Modifier.padding(4.dp, 8.dp), verticalAlignment = Alignment.CenterVertically) {
            val textSpec = textSpec(key)
            val baselineShiftModel = textInputModel(textSpec.baselineShift.multiplier.toString())
            val lineHeightModel = textInputModel(textSpec.lineHeight?.toString() ?: "0")
            val baselineShift = baselineShiftModel.state.read().content.toFloatOrNull()?.let {
              BaselineShift(it)
            } ?: BaselineShift.None
            val lineHeight = lineHeightModel.state.read().content.toFloatOrNull()
            val textSpecWithBaseline = textSpec.copy(baselineShift = baselineShift).letIfNotNull(lineHeight) { x, y ->
              x.copy(lineHeight = y)
            }
            UiText(text, userTextSpec = textSpecWithBaseline, backgroundColor = Color.LightGray.applyAlpha(0.3f))
            UiText("Baseline Shift:")
            Spacer(Modifier.width(4.dp))
            textInput(baselineShiftModel, style = innerTextInputStyle(), modifier = Modifier.weight(1f))
            Spacer(Modifier.width(4.dp))
            UiText("Line Height:")
            Spacer(Modifier.width(4.dp))
            textInput(lineHeightModel, style = innerTextInputStyle(), modifier = Modifier.weight(1f))
          }
        }
      }
    }
  }


  example("Emoji") {
    UiText("""😁🔥🥦👮🏼""")
  }

  example("Custom Color") {
    UiText("Colored text", color = Color.Cyan)
  }

  example("Custom Spec") {
    UiText("Bold text", textStyleKey = TextStyleKeys.DefaultSemiBold)
  }

  val loremIpsum = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum."

  example("Wrapping Text in a Box") {
    UiText(text = loremIpsum,
           modifier = Modifier.widthIn(max = 400.dp),
           paragraphStyle = NoriaParagraphStyle.multiline)
  }

  example("Multiline Text") {
    splitView(direction = SplitDirection.HORIZONTAL) {
      first {
        UiText(loremIpsum, paragraphStyle = NoriaParagraphStyle.multiline)
      }
      second {
        UiText(loremIpsum, paragraphStyle = NoriaParagraphStyle.multiline)
      }
    }
  }

  example("Different Line Height") {
    val text = "First line\nSecond line"
    val textSpec = textSpec(TextStyleKeys.Default)

    @Composable
    fun RowScope.withHeight(height: Float?) {
      Column(modifier = Modifier.padding(all = 4.dp)) {
        UiText("height: $height")
        Spacer(Modifier.width(4.dp))
        UiText(text,
               userTextSpec = textSpec.copy(lineHeight = height),
               paragraphStyle = NoriaParagraphStyle.multiline)
      }
    }
    Row {
      withHeight(null)
      withHeight(0f)
      withHeight(1f)
      withHeight(1.2f)
      withHeight(1.3f)
    }
  }

  example("Font Fallbacks") {
    val text = "Hello world こんにちは世界"
    Column {
      UiText(text, textStyleKey = TextStyleKeys.Code)
      UiText(text, textStyleKey = TextStyleKeys.Default, noriaTextDecoration = NoriaTextDecoration(Color.Red))
      UiText("שר-++ה", textStyleKey = TextStyleKeys.Code)
      UiText("שר-++ה", textStyleKey = TextStyleKeys.Default)
      UiText("Sarah (שר--+ה)", textStyleKey = TextStyleKeys.Code)
      UiText("Sarah (שר--+ה)", textStyleKey = TextStyleKeys.Default)
    }
  }

  example("Diacritics") {
    Column {
      UiText("Ex\u00E1mple (u00E1)", textStyleKey = TextStyleKeys.Header3SemiBold)
      UiText("Ex\u0061\u0301mple (u0061u0301)", textStyleKey = TextStyleKeys.Header3SemiBold)
    }
  }

  example("Different Font Rendering Settings") {
    // Now only low level text rendering api respects font rendering settings
    val theme = theme
    val textSpecState = state { theme.textSpec(TextStyleKeys.Code) }
    val fontSpec = textSpecState.read().fontSpec
    // defaults for my mac is:
    // FontHinting.NORMAL
    val hintingOptions = listOf(FontHinting.None, FontHinting.Slight, FontHinting.Normal, FontHinting.Full)
    // false
    val isAutoHintingForcedOptions = listOf(false, true)
    // false
    val subpixelOptions = listOf(false, true)
    // FontEdging.ANTI_ALIAS
    val edgingOptions = listOf(FontSmoothing.None, FontSmoothing.AntiAlias, FontSmoothing.SubpixelAntiAlias)
    val text = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua."
    Column {
      dropDownList(memo { listOf(textSpec(TextStyleKeys.Default), textSpec(TextStyleKeys.Code)) },
                   textSpecState,
                   presentationFn = { if (it == theme.textSpec(TextStyleKeys.Default)) "Default" else "Code" })
      for (subpixel in subpixelOptions) {
        for (edging in edgingOptions) {
          for (hinting in hintingOptions) {
            for (autoHintingForced in isAutoHintingForcedOptions) {
              val rasterSettings = FontRasterizationSettings(subpixelPositioning = subpixel,
                                                             smoothing = edging,
                                                             hinting = hinting,
                                                             autoHintingForced = autoHintingForced)
              CompositionLocalProvider(LocalFontRasterizationSettings provides rasterSettings) {
                UiText("$rasterSettings")
              }
              Spacer(Modifier.height(8.dp))
            }
          }
        }
      }
    }
  }

  example("Compose Doesn't Support Custom Decoration Color") {
    // we expect that underline will be transparent, but it's white apparently
    UiText("Hello!",
           color = Color.Black,
           modifier = Modifier,
           noriaTextDecoration = NoriaTextDecoration(Color.Green, style = NoriaTextDecoration.Style.SOLID, thicknessMultiplier = 1f),
           paragraphStyle = NoriaParagraphStyle.oneLine)
  }

  example("Text Code") {
    Column {
      UiText("Some Text!!!", textStyleKey = TextStyleKeys.Code, color = theme[ThemeKeys.TextSecondary])
      UiText("Some Unexpected Font", userTextSpec = TextSpec(FontSpec("Unexpected Font", 14.sp)))
    }
  }
}