@file:OptIn(ExperimentalFoundationApi::class)

package noria.ui.examples

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.layout.LazyLayout
import androidx.compose.foundation.lazy.layout.LazyLayoutItemProvider
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.BasicText
import androidx.compose.runtime.Composable
import androidx.compose.runtime.Stable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.clipToBounds
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.Placeable
import androidx.compose.ui.unit.Constraints
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import noria.NoriaContext
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery

private data class Item(val x: Int, val y: Int) {
  @Composable
  fun NoriaContext.render() {
    BasicText(
      text = "X: ${x}\nY: ${y}",
      color = { Color.White },
      modifier = Modifier
        .clip(RoundedCornerShape(24.dp))
        .background(Color.Cyan)
        .padding(12.dp)
    )
  }
}

private class MyItemProvider(val items: List<Item>) : LazyLayoutItemProvider {
  override val itemCount: Int
    get() = items.count()

  @Composable
  override fun NoriaContext.Item(index: Int) {
    with(items[index]) {
      render()
    }
  }

  fun getItemIndexesInRange(boundaries: ViewBoundaries): List<Int> {
    val result = mutableListOf<Int>()

    items.forEachIndexed { index, itemContent ->
      val listItem = itemContent
      if (listItem.x in boundaries.fromX..boundaries.toX &&
          listItem.y in boundaries.fromY..boundaries.toY
      ) {
        result.add(index)
      }
    }

    return result
  }

  fun getItem(index: Int): Item? {
    return items.getOrNull(index)
  }
}

@Composable
fun NoriaContext.rememberLazyLayoutState(): LazyLayoutState {
  return remember { LazyLayoutState() }
}

@Stable
class LazyLayoutState {

  val offsetState = mutableStateOf(IntOffset(0, 0))

  fun onDrag(offset: IntOffset) {
    val x = (offsetState.value.x - offset.x).coerceAtLeast(0)
    val y = (offsetState.value.y - offset.y).coerceAtLeast(0)
    offsetState.value = IntOffset(x, y)
  }

  fun getBoundaries(
    constraints: Constraints,
    threshold: Int = 500
  ): ViewBoundaries {
    return ViewBoundaries(
      fromX = offsetState.value.x - threshold,
      toX = constraints.maxWidth + offsetState.value.x + threshold,
      fromY = offsetState.value.y - threshold,
      toY = constraints.maxHeight + offsetState.value.y + threshold
    )
  }
}

data class ViewBoundaries(
  val fromX: Int,
  val toX: Int,
  val fromY: Int,
  val toY: Int
)

private fun Modifier.lazyLayoutPointerInput(state: LazyLayoutState): Modifier {
  return pointerInput(state) {
    detectDragGestures { change, dragAmount ->
      change.consume()
      state.onDrag(IntOffset(dragAmount.x.toInt(), dragAmount.y.toInt()))
    }
  }
}

private fun Placeable.PlacementScope.placeItem(state: LazyLayoutState, listItem: Item, placeables: List<Placeable>) {
  val xPosition = listItem.x - state.offsetState.value.x
  val yPosition = listItem.y - state.offsetState.value.y

  placeables.forEach { placeable ->
    placeable.placeRelative(
      xPosition,
      yPosition
    )
  }
}

@OptIn(ExperimentalFoundationApi::class)
fun composeLazyLayoutExamples(): Gallery = gallery("Compose Lazy Layout", NoriaExamples.sourceCodeForFile("ComposeLazyLayout.kt")) {
  //Inspired by: https://www.ackee.agency/blog/custom-lazylayout-with-jetpack-compose
  example("LazyLayout") {
    val items = List(10_000) {
      Item(x = (-20_000..20_000).random(),
           y = (-20_000..20_000).random())
    }
    val state = rememberLazyLayoutState()
    val itemProvider = MyItemProvider(items)
    Row(Modifier.width(500.dp).height(500.dp).background(Color.Gray)) {
      LazyLayout(itemProvider = itemProvider,
                 modifier = Modifier.clipToBounds().lazyLayoutPointerInput(state)
      ) { constraints ->
        val boundaries = state.getBoundaries(constraints)
        val indexes = itemProvider.getItemIndexesInRange(boundaries)

        val indexesWithPlaceables = indexes.associateWith {
          measure(it, Constraints())
        }

        layout(constraints.maxWidth, constraints.maxHeight) {
          indexesWithPlaceables.forEach { (index, placeables) ->
            val item = itemProvider.getItem(index)
            item?.let { placeItem(state, item, placeables) }
          }
        }
      }
    }
  }

  example("LazyColumn") {
    Row(Modifier.width(500.dp).height(500.dp).background(Color.Gray)) {
      LazyColumn {
        item {
          BasicText(text = "First Item  ")
        }
        items(100000) {
          BasicText(text = "Text No: $it ")
        }
        item {
          BasicText(text = "Last Item")
        }
      }
    }
  }
  example("Box with Constaints") {
    Column {
      BoxWithConstraints(modifier = Modifier.width(500.dp).height(500.dp).background(Color.Gray)) {
        BasicText(text = "Constraints = ${this.constraints}", modifier = Modifier.background(Color.Red).size(400.dp))

        Spacer(Modifier.align(Alignment.BottomStart).background(Color.Green).size(200.dp))
      }
    }
  }
}