package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.*
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import fleet.compose.foundation.text.selection.NoriaSelectionContainer
import fleet.compose.theme.components.UiText
import noria.NoriaContext
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.keys.TextStyleKeys
import fleet.compose.theme.text.NoriaParagraphStyle
import fleet.compose.theme.text.TextSpec
import fleet.compose.theme.theme
import kotlin.math.max
import kotlin.math.min

internal fun composeAlignmentLinesExamples(): Gallery = gallery("Compose Alignment Lines", NoriaExamples.sourceCodeForFile("ComposeAlignmentLines.kt")) {

  fun TextSpec.withFontSize(size: TextUnit): TextSpec {
    return this.copy(fontSpec = this.fontSpec.copy(size = size))
  }

  example("Simple Text Alignment Last Baseline") {
    Row {
      UiText(
        text = "One\nOne",
        userTextSpec = theme.textSpec(TextStyleKeys.Default).withFontSize(20.sp),
        paragraphStyle = NoriaParagraphStyle.multiline,
        modifier = Modifier.alignBy(LastBaseline).padding(10.dp).background(Color.Red)
      )

      UiText(
        text = "Two",
        userTextSpec = theme.textSpec(TextStyleKeys.Default).withFontSize(12.sp),
        modifier = Modifier.alignBy(LastBaseline).background(Color.Green)
      )
    }
  }

  example("Simple Text Alignment First Baseline") {
    Row {
      UiText(
        text = "One\nOne",
        userTextSpec = theme.textSpec(TextStyleKeys.Default).withFontSize(20.sp),
        paragraphStyle = NoriaParagraphStyle.multiline,
        modifier = Modifier.alignBy(FirstBaseline).padding(10.dp).background(Color.Red)
      )

      UiText(
        text = "Two",
        userTextSpec = theme.textSpec(TextStyleKeys.Default).withFontSize(12.sp),
        modifier = Modifier.alignBy(FirstBaseline).background(Color.Green)
      )
    }
  }

  example("Mixed Text Align") {
    Row {
      UiText(
        text = "One\nOne",
        userTextSpec = theme.textSpec(TextStyleKeys.Default).withFontSize(20.sp),
        paragraphStyle = NoriaParagraphStyle.multiline,
        modifier = Modifier.alignBy(FirstBaseline).background(Color.Red)
      )

      UiText(
        text = "Two\nTwo",
        userTextSpec = theme.textSpec(TextStyleKeys.Default).withFontSize(12.sp),
        paragraphStyle = NoriaParagraphStyle.multiline,
        modifier = Modifier.alignBy(LastBaseline).background(Color.Green)
      )
    }
  }
  example("Alignment Lines With SelectionContainer FL-27290") {
    NoriaSelectionContainer {
      Row {
        UiText("Text1", textStyleKey = TextStyleKeys.Default, modifier = Modifier.alignByBaseline())
        UiText("Text2", textStyleKey = TextStyleKeys.Header1, modifier = Modifier.alignByBaseline())
      }
    }
  }
  example("Simple Relative To Siblings With Offset") {
    Row(horizontalArrangement = Arrangement.spacedBy(4.dp)) {
      TestAlignmentWithOffset(Modifier.background(Color.LightGray))

      SimpleRelativeToSiblingsInColumn(Modifier.background(Color.Gray))
    }
  }
  example("Simple Relative To Siblings") {
    Column {
      SimpleRelativeToSiblings()
    }
  }
  example("Simple Relative To Siblings In Column") {
    Column {
      SimpleRelativeToSiblingsInColumn()
    }
  }
  example("Padding From Alignment Line To The Container") {
    Column {
      TestPaddingFromAlignmentLine()
    }
  }
}

@Composable
private fun NoriaContext.TestPaddingFromAlignmentLine() {
  val start = remember { HorizontalAlignmentLine(::min) }
  val middle = remember { HorizontalAlignmentLine(::min) }
  val end = remember { HorizontalAlignmentLine(::min) }

  @Composable
  fun NoriaContext.Example(modifier: Modifier = Modifier, color: Color, width: Dp, height: Dp) {
    Layout(
      content = { },
      modifier = modifier.background(color = color)
    ) { _, constraints ->
      val widthPx = max(width.roundToPx(), constraints.minWidth)
      val heightPx = max(height.roundToPx(), constraints.minHeight)
      layout(widthPx, heightPx, mapOf(start to 0, middle to widthPx / 2, end to widthPx)) {}
    }
  }
  Row(modifier = Modifier.background(Color.LightGray), horizontalArrangement = Arrangement.spacedBy(4.dp)) {
    Example(modifier = Modifier.alignBy(end),
            color = Color.Red,
            width = 50.dp,
            height = 50.dp)
    Example(modifier = Modifier.alignBy(middle),
            color = Color.Green,
            width = 50.dp,
            height = 50.dp)
    // makes 125.dp padding from the top of the container to the `middle` alignment and 125.dp from the bottom to the `middle`
    Example(modifier = Modifier.paddingFrom(middle, 125.dp, 125.dp),
            color = Color.Cyan,
            width = 50.dp,
            height = 50.dp)
    Example(modifier = Modifier.alignBy(start),
            color = Color.Blue,
            width = 50.dp,
            height = 50.dp)
  }
}


@Composable
private fun NoriaContext.SimpleRelativeToSiblingsInColumn(modifier: Modifier = Modifier) {
  // Alignment lines provided by the RectangleWithStartEnd layout. We need to create these
  // local alignment lines because Compose is currently not providing any top-level out of
  // the box vertical alignment lines. Two horizontal alignment lines are provided though:
  // FirstBaseline and LastBaseline, but these can only be used to align vertically children
  // of Row because the baselines are horizontal. Therefore, we create these vertical alignment
  // lines, that refer to the start and end of the RectangleWithStartEnd layout which knows
  // how to provide them. Then Column will know how to align horizontally children such
  // that the positions of the alignment lines coincide, as asked by alignBy.

  val start = remember { VerticalAlignmentLine(::min) }
  val end = remember { VerticalAlignmentLine(::min) }

  @Composable
  fun NoriaContext.RectangleWithStartEnd(modifier: Modifier = Modifier, color: Color, width: Dp, height: Dp) {
    Layout(
      content = { },
      modifier = modifier.background(color = color)
    ) { _, constraints ->
      val widthPx = max(width.roundToPx(), constraints.minWidth)
      val heightPx = max(height.roundToPx(), constraints.minHeight)
      layout(widthPx, heightPx, mapOf(start to 0, end to widthPx)) {}
    }
  }

  Column(modifier) {
    // Center of the first rectangle is aligned to the right edge of the second rectangle and
    // left edge of the third one.
    Box(
      Modifier
        .size(90.dp, 45.dp)
        .alignBy { it.measuredWidth / 2 }
        .background(Color.Blue)
    )
    RectangleWithStartEnd(
      Modifier.alignBy(end),
      color = Color.Magenta,
      width = 80.dp,
      height = 40.dp
    )
    RectangleWithStartEnd(
      Modifier.alignBy(start),
      color = Color.Red,
      width = 100.dp,
      height = 50.dp
    )
  }
}

@Composable
private fun NoriaContext.SimpleRelativeToSiblings() {
  Column {
    // Center of the first rectangle is aligned to the right edge of the second rectangle and
    // left edge of the third one.
    Box(
      Modifier.size(80.dp, 40.dp)
        .alignBy { it.measuredWidth / 2 }
        .background(Color.Blue)
    )
    Box(
      Modifier.size(80.dp, 40.dp)
        .alignBy { it.measuredWidth }
        .background(Color.Magenta)
    )
    Box(
      Modifier.size(80.dp, 40.dp)
        .alignBy { 0 }
        .background(Color.Red)
    )
  }
}


@Composable
private fun NoriaContext.TestAlignmentWithOffset(modifier: Modifier = Modifier) {
  val start = remember { VerticalAlignmentLine(::min) }
  val end = remember { VerticalAlignmentLine(::min) }

  @Composable
  fun NoriaContext.RectangleWithStartEnd(modifier: Modifier = Modifier, color: Color, width: Dp, height: Dp) {
    Layout(
      content = { },
      modifier = modifier.background(color = color)
    ) { _, constraints ->
      val widthPx = max(width.roundToPx(), constraints.minWidth)
      val heightPx = max(height.roundToPx(), constraints.minHeight)
      layout(widthPx, heightPx, mapOf(start to 0, end to widthPx)) {}
    }
  }

  Column(modifier) {
    Box(
      Modifier
        .offset(5.dp, 5.dp)
        .size(90.dp, 45.dp)
        .alignBy { it.measuredWidth / 2 }
        .background(Color.Blue)
    )
    RectangleWithStartEnd(
      Modifier.offset(6.dp, 6.dp).alignBy(end),
      color = Color.Magenta,
      width = 80.dp,
      height = 40.dp
    )
    RectangleWithStartEnd(
      Modifier.offset(7.dp, 7.dp).alignBy(start),
      color = Color.Red,
      width = 100.dp,
      height = 50.dp
    )
  }
}