(function() {
  const obtainPositionAttributeName = () => {
    const element = document.querySelector(`meta[name="markdown-position-attribute-name"]`);
    return element.content;
  };

  const performSmoothScroll = (element) => {
    return new Promise( (resolve) => {
      let frames = 0;
      let lastPosition = null;
      element.scrollIntoView({ behavior: "smooth" });
      const action = () => {
        const currentPosition = element.getBoundingClientRect().top;
        if (currentPosition === lastPosition) {
          frames += 1;
          if (frames > 2) {
            return resolve();
          }
        } else {
          frames = 0;
          lastPosition = currentPosition;
        }
        requestAnimationFrame(action);
      };
      requestAnimationFrame(action);
    });
  };

  const findMainContainer = () => {
    const selector = "main#main-content"
    const elements = document.querySelectorAll(selector);
    if (elements.length === 0) {
      throw new Error(`There is no ${selector} element on the current page`);
    }
    if (elements.length > 1) {
      throw new Error(`There is more than one ${selector} element`);
    }
    return elements.item(0);
  };

  const findRootElementOrNull = () => {
    const main = findMainContainer();
    return main.firstChild || null;
  };

  const findRootElement = () => {
    const root = findRootElementOrNull();
    if (!root) {
      throw new Error("Could not find root element as a first child of main");
    }
    return root;
  };

  class ScrollManager {
    #positionAttributeName = obtainPositionAttributeName();
    #scrollFinished = true;
    #currentScrollElement = null;
    #lastOffset = 0;
    #collectMarkdownElements = () => [];

    constructor() {
      this.#updateMarkdownElements();
    }

    #textRangeOf(node) {
      if (!node || !("getAttribute" in node)) {
        return null;
      }
      const value = node.getAttribute(this.#positionAttributeName);
      if (value) {
        return value.split("..");
      }
      return null;
    }

    #obtainMaxPossibleOffset() {
      const element = findRootElement();
      const offsets = this.#textRangeOf(element);
      if (!offsets) {
        throw new Error("Root element should have defined offsets");
      }
      return offsets[1];
    }

    #updateMarkdownElements() {
      this.#collectMarkdownElements = this.#doCollectMarkdownElements();
    }

    #doCollectMarkdownElements() {
      let elements = null;
      return () => {
        if (elements != null) {
          return elements;
        }
        elements = Array.from(document.querySelectorAll(`[${this.#positionAttributeName}]`)).map(element => {
          const position = element.getAttribute(this.#positionAttributeName).split("..");
          return {
            element,
            from: position[0],
            to: position[1]
          };
        });
        return elements;
      };
    }

    #doScroll(element, smooth) {
      if (!smooth) {
        element.scrollIntoView();
        return;
      }
      this.#scrollFinished = false;
      performSmoothScroll(element).then(() => {
        this.#scrollFinished = true;
      });
    }

    scrollToOffset(offset, smooth = true) {
      if (this.currentScrollElement) {
        const position = this.#textRangeOf(this.currentScrollElement);
        if (!position) {
          throw new Error("Current scroll element should have valid associated text range");
        }
        const [startOffset, endOffset] = position;
        if (offset >= startOffset && offset <= endOffset) {
          return;
        }
      }
      const root = findRootElementOrNull();
      if (!root) {
        return;
      }
      const element = this.#actuallyFindElement(root, offset, offset >= this.#lastOffset);
      this.#lastOffset = offset;
      if (!element) {
        console.warn(`Failed to find element for offset: ${offset}`);
        return;
      }
      this.currentScrollElement = element;
      this.#doScroll(element, smooth);
    }

    scrollBy(horizontal, vertical) {
      if (this.#scrollFinished) {
        window.scrollBy(horizontal, vertical);
      }
    }

    #findElementAtOffset(offset, node, result = {}) {
      for (let child = node.firstChild; child !== null; child = child.nextSibling) {
        // if (child.nodeType !== Node.ELEMENT_NODE) {
        //   continue;
        // }
        const position = this.#textRangeOf(child);
        if (!position) {
          continue;
        }
        const [startOffset, endOffset] = position;
        if (offset >= startOffset && offset <= endOffset) {
          result.element = child;
          this.#findElementAtOffset(offset, child, result);
          break;
        }
      }
      return result.element;
    }

    #actuallyFindElement(root, offset, forward = false) {
      const targetElement = this.#findElementAtOffset(offset, root);
      if (targetElement) {
        return targetElement;
      }
      if (forward) {
        const maxOffset = this.#obtainMaxPossibleOffset();
        for (let it = offset; it <= maxOffset; it += 1) {
          const previousElement = this.#findElementAtOffset(it, root);
          if (previousElement) {
            return previousElement;
          }
        }
      } else {
        for (let it = offset - 1; it >= 0; it -= 1) {
          const previousElement = this.#findElementAtOffset(it, root);
          if (previousElement) {
            return previousElement;
          }
        }
      }
      return null;
    }
  }

  const scrollManager = new ScrollManager();
  window["obtainScrollManager"] = () => {
    return scrollManager;
  };
})();
