package noria.ui.examples

import kotlinx.coroutines.delay
import noria.*
import androidx.compose.runtime.remember
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import noria.ui.components.*
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.gallery.Gallery
import noria.ui.core.*
import fleet.compose.theme.components.UiText
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.dp
import fleet.compose.theme.launchRestart
import noria.ui.withModifier
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.DurationUnit

internal fun enhance(): Gallery = gallery("Enhance!",
                                          NoriaExamples.sourceCodeForFile("Enhance.kt")) {

  example("Enhance") {
    val maxSubdivisionDepthInputModel = textInputModel("6")
    val subdivisionIntervalInputModel = textInputModel("10")
    val maxSubdivisionDepth = maxSubdivisionDepthInputModel.state.read().content.toIntOrNull() ?: 6
    val interval = subdivisionIntervalInputModel.state.read().content.toIntOrNull()?.milliseconds ?: 10.milliseconds
    val currentSubdivisionDepth = state(maxSubdivisionDepth) { 0 }
    val boxHueState = state { 0f }

    launchRestart(maxSubdivisionDepth, interval) {
      var delta = 1
      while (true) {
        delay(interval)
        when (currentSubdivisionDepth.readNonReactive()) {
          maxSubdivisionDepth -> delta = -1
          0 -> {
            delta = 1
            boxHueState.update { (it + interval.toDouble(DurationUnit.SECONDS).toFloat()) % 1f }
          }
        }
        currentSubdivisionDepth.update { (it + delta).coerceIn(0, maxSubdivisionDepth) }
      }
    }

    Column {
      Row(verticalAlignment = Alignment.CenterVertically) {
        UiText("Maximum Subdivision Depth")
        Spacer(Modifier.width(4.dp))
        textInput(maxSubdivisionDepthInputModel)
        Spacer(Modifier.width(32.dp))
        UiText("Subdivision Interval")
        Spacer(Modifier.width(4.dp))
        textInput(subdivisionIntervalInputModel, suffixBuilder = { UiText("ms") })
      }
      Spacer(Modifier.height(4.dp))
      Row {
        CompositionLocalProvider(LocalBoxHueCell provides boxHueState) {
          boundary {
            withModifier(Modifier.size(512.dp, 512.dp)) {
    withFpsOverlay {
    recursivelySplittingBox(DpSize(512.dp, 512.dp), 0, currentSubdivisionDepth.read())
  }
            }
          }
        }
      }
    }
  }
}


@Composable
private fun NoriaContext.recursivelySplittingBox(size: DpSize, depth: Int, targetDepth: Int) {
  boundary {
    if (depth == targetDepth) {
      val boxHueState = LocalBoxHueCell.current
      val color = remember(boxHueState) { randomBrightColor(boxHueState.read()) }
      Spacer(Modifier.background(color).size(size.width, size.height))
    }
    else {
      Column {
        Row {
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth)
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth)
        }
        Row {
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth)
          recursivelySplittingBox(DpSize(size.width / 2, size.height / 2), depth + 1, targetDepth)
        }
      }
    }
  }
}
