package noria.ui.examples


import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.BasicText
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import fleet.compose.foundation.culling.layout.CullingLayout
import fleet.compose.theme.Edge
import fleet.compose.theme.LaunchedBackgroundEffect
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.keys.ThemeKeys
import fleet.compose.theme.layout.outset
import fleet.compose.theme.outline
import fleet.compose.theme.surface
import fleet.compose.theme.text.NoriaParagraphStyle
import fleet.compose.theme.theme
import fleet.compose.theme.ui.largeShadow
import fleet.compose.theme.ui.mediumShadow
import fleet.compose.theme.ui.smallShadow
import fleet.util.EnumSet
import fleet.util.enumSetOfAll
import kotlinx.coroutines.delay
import noria.ui.components.textInput
import noria.ui.components.textInputModel
import kotlin.time.Duration.Companion.seconds

internal fun composeModifiersExamples(): Gallery = gallery("Compose Modifiers", NoriaExamples.sourceCodeForFile("ComposeModifiers.kt")) {
  example("Compose Shadow Example") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp), modifier = Modifier.padding(all = 20.dp)) {
      Row(horizontalArrangement = Arrangement.spacedBy(50.dp)) {
        val cornerSize = 4.dp
        val shape = RoundedCornerShape(cornerSize)
        val size = 100.dp

        Box(contentAlignment = Alignment.Center) {
          Spacer(
            Modifier
              .background(Color.Yellow)
              .size(size * 2)
          )

          Spacer(
            Modifier
              .largeShadow(shape)
              .clip(shape)
              .background(Color.Blue)
              .size(size)
          )
        }

        Spacer(
          Modifier
            .smallShadow(shape)
            .background(theme[ThemeKeys.TextPrimary], shape)
            .size(size)
        )

        Spacer(
          Modifier
            .mediumShadow(shape)
            .background(theme[ThemeKeys.TextPrimary], shape)
            .size(size)
        )

        Spacer(
          Modifier
            .largeShadow(shape)
            .background(theme[ThemeKeys.TextPrimary], shape)
            .size(size)
        )
      }
    }
  }

  example("Compose Outset") {
    Box(
      Modifier
        .border(1.dp, Color.Blue)
        .padding(5.dp)
        .border(1.dp, Color.Green)
        .outset(5.dp)
        .background(Color.Red)
        .size(100.dp)
    )
  }
  example("Compose Outline Example") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp)) {
      val edges = remember { enumSetOfAll<Edge>().toMutableStateList() }
      var topCornerRadius by remember { mutableStateOf(10.dp) }
      var bottomCornerRadius by remember { mutableStateOf(4.dp) }
      Row(verticalAlignment = Alignment.CenterVertically) {
        textInput("${topCornerRadius.value}", onInput = {
          it.toFloatOrNull()?.let { topCornerRadius = it.dp }
        })
        BasicText("Top Corner Radius")
      }
      Row(verticalAlignment = Alignment.CenterVertically) {
        textInput("${bottomCornerRadius.value}", onInput = {
          it.toFloatOrNull()?.let { bottomCornerRadius = it.dp }
        })
        BasicText("Bottom Corner Radius")
      }
      Row(horizontalArrangement = Arrangement.spacedBy(10.dp)) {
        Checkbox(
          Edge.Top in edges,
          onCheckedChange = { if (it) edges.add(Edge.Top) else edges.remove(Edge.Top) },
          label = "Top"
        )
        Checkbox(
          Edge.Bottom in edges,
          onCheckedChange = { if (it) edges.add(Edge.Bottom) else edges.remove(Edge.Bottom) },
          label = "Bottom"
        )
        Checkbox(
          Edge.Start in edges,
          onCheckedChange = { if (it) edges.add(Edge.Start) else edges.remove(Edge.Start) },
          label = "Start"
        )
        Checkbox(
          Edge.End in edges,
          onCheckedChange = { if (it) edges.add(Edge.End) else edges.remove(Edge.End) },
          label = "End"
        )
      }
      Row(horizontalArrangement = Arrangement.spacedBy(10.dp)) {
        val borderWidth = 4.dp
        val size = 80.dp
        val outlineColor = Color.Red
        val borderColor = Color.Cyan
        val backgroundColor = Color.Magenta
        val edgeSet = if (edges.isEmpty()) EnumSet.noneOf<Edge>() else EnumSet.copyOf(edges)

        val shape = RoundedCornerShape(topCornerRadius, topCornerRadius, bottomCornerRadius, bottomCornerRadius)

        Spacer(
          Modifier
            .outline(borderWidth, outlineColor, shape, edgeSet)
            .surface(backgroundColor, borderWidth, borderColor, shape, edgeSet)
            .size(size)
        )

        Spacer(
          Modifier
            .outline(borderWidth, outlineColor, shape, edgeSet)
            .background(backgroundColor, shape)
            .size(size)
        )
      }
    }
  }
  example("Compose Surface Example") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp)) {
      val edges = remember { enumSetOfAll<Edge>().toMutableStateList() }
      var topCornerRadius by remember { mutableStateOf(10.dp) }
      var bottomCornerRadius by remember { mutableStateOf(4.dp) }
      Row(verticalAlignment = Alignment.CenterVertically) {
        textInput("${topCornerRadius.value}", onInput = {
          it.toFloatOrNull()?.let { topCornerRadius = it.dp }
        })
        BasicText("Top Corner Radius")
      }
      Row(verticalAlignment = Alignment.CenterVertically) {
        textInput("${bottomCornerRadius.value}", onInput = {
          it.toFloatOrNull()?.let { bottomCornerRadius = it.dp }
        })
        BasicText("Bottom Corner Radius")
      }
      Row(horizontalArrangement = Arrangement.spacedBy(10.dp)) {
        Checkbox(
          Edge.Top in edges,
          onCheckedChange = {
            if (Edge.Top in edges) edges.remove(Edge.Top) else edges.add(Edge.Top)
          },
          label = "Top"
        )
        Checkbox(
          Edge.Bottom in edges,
          onCheckedChange = { if (Edge.Bottom in edges) edges.remove(Edge.Bottom) else edges.add(Edge.Bottom) },
          label = "Bottom"
        )
        Checkbox(
          Edge.Start in edges,
          onCheckedChange = { if (Edge.Start in edges) edges.remove(Edge.Start) else edges.add(Edge.Start) },
          label = "Start"
        )
        Checkbox(
          Edge.End in edges,
          onCheckedChange = { if (Edge.End in edges) edges.remove(Edge.End) else edges.add(Edge.End) },
          label = "End"
        )
      }
      Row(horizontalArrangement = Arrangement.spacedBy(10.dp)) {
        val borderWidth = 4.dp
        val size = 80.dp
        val borderColor = Color.Cyan
        val backgroundColor = Color.Magenta
        val edgeSet = if (edges.isEmpty()) EnumSet.noneOf<Edge>() else EnumSet.copyOf(edges)

        val shape = RoundedCornerShape(topCornerRadius, topCornerRadius, bottomCornerRadius, bottomCornerRadius)

        Spacer(Modifier.border(borderWidth, borderColor, shape).background(backgroundColor, shape).size(size))

        Spacer(Modifier.surface(backgroundColor, borderWidth, borderColor, shape, edgeSet).size(size))
      }
    }
  }

  example("Compose Conditional Modifiers") {
    Column {
      UiText(
        "The Box around the text field below will switch between having and not having modifiers every second. It may not lose its state when that happens. Try typing into it and check if the caret blinks normally!",
        paragraphStyle = NoriaParagraphStyle.multiline,
      )

      var shouldSetModifiers by remember { mutableStateOf(true) }
      Box {
        Box(if (shouldSetModifiers) {
          Modifier
            .border(3.dp, Color.Magenta)
            .drawBehind { drawRect(Color.Green, alpha = 0.3f) }
        }
            else {
          Modifier
        }) {
          remember { println("Composition identity lost") }
          //Box(Modifier.background(Color.Red).size(100.dp))
          textInput(textInputModel())
          CullingLayout { constraints ->
            layout(constraints.minWidth, constraints.minHeight) {
              remember { println("Viewport identity lost") }
            }
          }
        }
      }
      LaunchedBackgroundEffect(Unit) {
        while (true) {
          delay(1.seconds)
          shouldSetModifiers = !shouldSetModifiers
        }
      }
    }
  }
}
