package noria.ui.examples

import androidx.compose.animation.core.*
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.Layout
import androidx.compose.ui.unit.dp
import androidx.compose.ui.zIndex
import fleet.compose.theme.components.Button
import fleet.compose.theme.components.checkbox.Checkbox
import noria.NoriaContext
import fleet.compose.theme.components.Thickness
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import noria.ui.core.boundary
import fleet.compose.theme.components.UiText
import noria.ui.withModifier

@Composable
private fun NoriaContext.CustomLayout(
  modifier: Modifier = Modifier,
  content: @Composable NoriaContext.() -> Unit
) {
  Layout(
    modifier = modifier,
    content = { content() }
  ) { measurables, constraints ->
    // Don't constrain child views further, measure them with given constraints
    // List of measured children
    val placeables = measurables.map { measurable ->
      // Measure each children
      measurable.measure(constraints)
    }

    // Set the size of the layout as big as it can
    layout(constraints.maxWidth, constraints.maxHeight) {
      // Track the y co-ord we have placed children up to
      var yPosition = 0

      // Place children in the parent layout
      placeables.forEach { placeable ->
        // Position item on the screen
        placeable.placeRelative(x = 0, y = yPosition)

        // Record the y co-ord placed up to
        yPosition += placeable.height
      }
    }
  }
}

internal fun composeLayoutExamples(): Gallery = gallery("Compose Layout", NoriaExamples.sourceCodeForFile("ComposeLayout.kt")) {
  example("Basic Custom Layout") {
    Row {
      CustomLayout(
        Modifier
          .sizeIn(maxWidth = 200.dp, maxHeight = 500.dp, )
          .background(Color.Green),
      ) {
        //withModifier(Modifier.background(Color.Magenta, shape = AbsoluteRoundedCornerShape(4.unit))) {
        //  gap(10.unit, 10.unit)
        //}
        UiText("MyBasicColumn")
        withModifier(Modifier.size(width = 20.dp,
                                   height = 50.dp)) {
          Spacer(Modifier.background(Color.Blue))
        }
        UiText("places items")
        UiText("vertically.")
        UiText("We've done it by hand!")
      }
    }
  }
  example("Compose Offset Layout Modifier") {
    Column {
      withModifier(Modifier.size(width = 200.dp, height = 500.dp)) {
    val transition = rememberInfiniteTransition()
 val offsetX by transition.noriaAnimateFloat(this, 20f, 180f.dp.value,
                                             infiniteRepeatable(animation = tween(1000),
                                                           repeatMode = RepeatMode.Reverse))
 val offsetY = transition.noriaAnimateFloat(this, 20f, 460.dp.value,
                                            infiniteRepeatable(animation = tween(1000),
                                                          repeatMode = RepeatMode.Reverse)).value
 UiText("hello world", modifier = Modifier.offset(offsetX.dp, offsetY.dp))
      }
    }
  }
  example("Compose Padding Layout Modifier") {
    Column {
      Spacer(Modifier
               .background(color = Color.Green)
               .padding(all = 10.dp)
               .border(width = Thickness.Regular, Color.Black)
               .padding(all = 30.dp)
               .border(width = Thickness.Regular, Color.Red)
               .padding(all = 80.dp).size(40.dp, 40.dp))
    }
  }

  for (horizontal in listOf(Alignment.Start, Alignment.CenterHorizontally, Alignment.End)) {
    for (vertical in listOf(Arrangement.Bottom,
                            Arrangement.Center,
                            Arrangement.Top,
                            Arrangement.SpaceEvenly,
                            Arrangement.SpaceBetween)) {
      example("Compose Column, Arrangement: $vertical Alignment: $horizontal") {
        withModifier(Modifier.size(width = 160.dp, height = 160.dp)) {
    Column(verticalArrangement = vertical,
              horizontalAlignment = horizontal,
              modifier = Modifier.background(Color.LightGray)) {
    UiText("Line1")
    UiText("Line2")
    UiText("Line3")
  }
        }
      }
    }
  }

  for (vertical in listOf(Alignment.Top, Alignment.CenterVertically, Alignment.Bottom)) {
    for (horizontal in listOf(Arrangement.Start,
                              Arrangement.Center,
                              Arrangement.End,
                              Arrangement.SpaceEvenly,
                              Arrangement.SpaceBetween)) {
      example("Compose Row, Arrangement: $horizontal Alignment: $vertical") {
        withModifier(Modifier.size(width = 160.dp, height = 160.dp)) {
            Row(horizontalArrangement = horizontal,
                   verticalAlignment = vertical,
                   modifier = Modifier.background(Color.LightGray)) {
            UiText("Line1")
            UiText("Line2")
            UiText("Line3")
          }
        }
      }
    }
  }

  example("Box Layout Example") {
    withModifier(Modifier.size(width = 160.dp, height = 160.dp)) {
    Box {
    UiText("This text is drawn first", modifier = Modifier.align(Alignment.TopCenter))
    Box(
      Modifier.align(Alignment.TopCenter).background(Color.Blue).padding(all = 40.dp)
    )
    UiText("This text is drawn last", modifier = Modifier.align(Alignment.Center))
    withModifier(Modifier.align(Alignment.BottomEnd).padding(12.dp)) {
      Button(text = "Button!", onClick = {})
    }
  }
    }
  }

  example("Propagate Parent Data Example") {
    withModifier(Modifier.size(width = 160.dp, height = 160.dp)) {
    Column(verticalArrangement = Arrangement.SpaceEvenly,
              horizontalAlignment = Alignment.CenterHorizontally,
              modifier = Modifier.background(Color.LightGray)) {
    UiText(
          "Color.Gray Alignment.Start",
      modifier = Modifier.background(Color.Gray).align(Alignment.Start),
    )

    UiText(
          "Alignment.Start Color.Cyan",
      modifier = Modifier.align(Alignment.Start).background(Color.Cyan),
    )

    UiText("Alignment.CenterHorizontally")
  }
    }
  }

  example("Fill Modifier Example") {
    Column {
      Spacer(Modifier.background(Color.Red).fillMaxWidth().height(40.dp))
    }
  }

  example("Aspect Ratio Example") {
    Column {
      Box(Modifier.width(100.dp).aspectRatio(2f).background(Color.Green))
    }
  }

  example("Layout ZIndex") {
    Column {
      var enabled by remember { mutableStateOf(true) }
      Checkbox(enabled, label = "Enable zIndex", onCheckedChange = {
        enabled = !it
      })
      Spacer(Modifier.height(4.dp))
      Row(Modifier.background(Color.LightGray)) {
        UiText(
          "Drawn second",
          modifier = Modifier
            .zIndex(if (enabled) 1f else 0f)
            .offset(x = 20.dp)
            .background(Color.Blue)
        )

        UiText(
          "Drawn first",
          modifier = Modifier
            .background(Color.Red)
        )
      }
    }
  }

  example("Layout ZIndex With Boundary") {
    Column {
      var enabled by remember { mutableStateOf(true) }
      Checkbox(enabled, label = "Enable zIndex", onCheckedChange = {
        enabled = !it
      })
      Spacer(Modifier.height(4.dp))
      boundary {
        Row(Modifier.background(Color.LightGray)) {
          boundary {
            UiText(
              "Drawn second",
              modifier = Modifier
                .zIndex(if (enabled) 1f else 0f)
                .offset(x = 20.dp)
                .background(Color.Blue)
            )
          }
          boundary {
            UiText(
              "Drawn first",
              modifier = Modifier
                .background(Color.Red)
            )
          }
        }
      }
    }
  }

  example("Layout ZIndex Summing") {
    Column {
      var enabled by remember { mutableStateOf(true) }
      Checkbox(enabled, label = "Enable zIndex", onCheckedChange = {
        enabled = !it
      })
      Spacer(Modifier.height(4.dp))
      Row(Modifier.background(Color.LightGray)) {
        UiText(
          "Drawn second",
          modifier = Modifier
            .zIndex(if (enabled) 2f else 0f)
            .offset(x = 20.dp)
            .zIndex(2f)
            .background(Color.Blue)
        )

        UiText(
          "Drawn first",
          modifier = Modifier
            .zIndex(3f)
            .background(Color.Red)
        )
      }
    }
  }
}
