package noria.ui.examples

import androidx.compose.animation.*
import androidx.compose.animation.core.*
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.BasicText
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.asComposePaint
import androidx.compose.ui.graphics.asComposePath
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.dp
import fleet.compose.theme.components.Button
import fleet.compose.theme.components.Thickness
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.buttonStyle
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.graphics.applyAlpha
import noria.scope
import noria.state
import noria.ui.components.dropDownList
import noria.ui.core.animation.animateValueOnce
import fleet.compose.theme.toPaint
import noria.ui.withModifier
import org.jetbrains.skia.Matrix33
import org.jetbrains.skia.Path
import org.jetbrains.skia.impl.use

internal fun animationExamples() = gallery("Animations", NoriaExamples.sourceCodeForFile("Animations.kt")) {
  example("Crossfade Animation") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp)) {
      var isVisible by remember { mutableStateOf(true) }
      Checkbox(isVisible, onCheckedChange = { isVisible = it }, label = "Is Visible")

      Crossfade(isVisible) { visible ->
        if (visible) {
          BasicText("Visible")
        }
        else {
          BasicText("Not Visible")
        }
      }
    }
  }

  example("Crossscale Animation") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp)) {
      var isVisible by remember { mutableStateOf(true) }
      Checkbox(isVisible, onCheckedChange = { isVisible = it }, label = "Is Visible")

      Crossscale(isVisible) { visible ->
        if (visible) {
          BasicText("Visible")
        }
        else {
          BasicText("Not Visible")
        }
      }
    }
  }

  example("Animate Visibility. Default Transitions") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp)) {
      var isVisible by remember { mutableStateOf(true) }
      Checkbox(isVisible, label = if (isVisible) "Visible" else "Hidden", onCheckedChange = {
        isVisible = !it
      })

      AnimatedVisibility(isVisible) {
        Box(
          Modifier
            .background(Color.Red.applyAlpha(.5f))
            .size(40.dp)
        )
      }
    }
  }

  example("Animate Visibility. Fade In - Fade Out") {
    Column(verticalArrangement = Arrangement.spacedBy(20.dp)) {
      var isVisible by remember { mutableStateOf(true) }
      Checkbox(isVisible, label = if (isVisible) "Visible" else "Hidden", onCheckedChange = {
        isVisible = !it
      })

      AnimatedVisibility(isVisible, enter = fadeIn(), exit = fadeOut()) {
        Box(
          Modifier
            .background(Color.Red.applyAlpha(.5f))
            .size(40.dp)
        )
      }
    }
  }


  example("Explore Animation Easing") {
    val flag = state { true }
    val easings = EasingInformation
    val numberOfColumns = 3
    val widths = easings.withIndex().map { (offset, it) ->
      scope(offset) {
        animateDpAsState(if (flag.read()) 20.dp else 100.dp, tween(1000, easing = it.second))
      }
    }

    Column(horizontalAlignment = Alignment.CenterHorizontally) {
      UiText("Boxes is on the ${if (flag.read()) "left" else "right"}")
      Spacer(Modifier.height(8.dp))
      Button(text = "Move to ${if (flag.read()) "right" else "left"}", onClick = { flag.update { !it } })
      Spacer(Modifier.height(20.dp))
      for ((rowOffset, row) in (widths zip easings.map { it.first }).chunked(numberOfColumns).withIndex()) {
        scope(rowOffset) {
          Row(verticalAlignment = Alignment.CenterVertically) {
            Spacer(Modifier.width(20.dp))
            for ((offset, info) in row.withIndex()) {
              scope(offset) {
                withModifier(Modifier.width(width = 260.dp)) {
                  Column {
                    UiText(info.second)
                    Row {
                      Spacer(Modifier.width(info.first.value))
                      Box(
                        Modifier
                          .background(Color.Red.applyAlpha(.5f))
                          .size(size = 40.dp)
                      )
                    }
                  }
                }
              }
              Spacer(Modifier.width(8.dp))
            }
          }
        }
      }
    }
  }

  example("Explore Easing Curves") {
    Column {
      val cubicEasings = remember {
        EasingInformation.mapNotNull { info -> (info.second as? CubicBezierEasing)?.let { info.first to it } }
      }
      val easingState = state { cubicEasings.first() }
      withModifier(Modifier.width(width = 200.dp)) {
        dropDownList(cubicEasings, selectedItemState = easingState) { it.first }
      }
      Spacer(Modifier.height(24.dp))
      val dpSize = DpSize(160.dp, 80.dp)

      val easing = (easingState.read()).second
      val progress by animateValueOnce(
        easing,
        initialValue = 0f,
        targetValue = 1f,
        typeConverter = Float.VectorConverter,
        animationSpec = infiniteRepeatable(
          tween(durationMillis = 1500, easing = LinearEasing),
          repeatMode = RepeatMode.Reverse
        )
      )
      val radius = 4.dp
      Box(modifier = Modifier.wrapContentSize(Alignment.Center).size(dpSize), contentAlignment = Alignment.Center) {
        Canvas(Modifier.fillMaxSize()) {
          Path()
            .moveTo(0f, 1f)
            .cubicTo(easing.a, 1 - easing.b, easing.c, 1 - easing.d, 1f, 0f)
            .transform(Matrix33.makeScale(size.width, size.height))
            .use { path ->
              Color.Black.toPaint().use { paint ->
                paint.strokeWidth = 2f
                paint.setStroke(true)
                drawIntoCanvas { canvas ->
                  canvas.drawPath(path.asComposePath(), paint.asComposePaint())
                }
              }
            }

          Path()
            .addCircle(0f, size.height, radius.roundToPx().toFloat())
            .transform(Matrix33.makeTranslate(size.width * progress, -easing.transform(progress) * size.height))
            .use { circle ->
              Color.Red.toPaint().use { paint ->
                drawIntoCanvas { canvas ->
                  canvas.drawPath(circle.asComposePath(), paint.asComposePaint())
                }
              }
            }
        }

        Spacer(Modifier.align(Alignment.TopStart).background(Color.Gray.applyAlpha(0.5f)).fillMaxHeight().width(Thickness.Regular))

        Spacer(Modifier.align(Alignment.BottomStart).background(Color.Gray.applyAlpha(0.5f)).fillMaxWidth().height(Thickness.Regular))
      }
    }
  }

  example("Basic Animation As State") {
    val flag = state { true }
    val width = animateDpAsState(if (flag.read()) 20.dp else 100.dp, spring(Spring.DampingRatioMediumBouncy))
    val color = animateColorAsState(if (flag.read()) Color.Black else Color.Red, tween(1000))

    Column {
      UiText("Box is on the ${if (flag.read()) "left" else "right"}")
      Spacer(Modifier.height(8.dp))
      Button(text = "Move to ${if (flag.read()) "right" else "left"}", onClick = { flag.update { !it } })
      Spacer(Modifier.height(20.dp))
      Row {
        Spacer(Modifier.width(width.value))
        Box(
          Modifier
            .background(color.value.applyAlpha(.5f))
            .size(40.dp)
        )
      }
    }
  }

  example("Complex Animation") {
    val isPressed = state { false }
    val background by animateColorAsState(if (isPressed.read()) Color.Red else Color.White, tween(1000, easing = FastOutSlowInEasing))
    val radius by animateDpAsState(if (isPressed.read()) 40.dp else 8.dp, tween(700, delayMillis = 300))
    val alpha by animateFloatAsState(if (isPressed.read()) 0.0f else 1.0f, tween(1000))

    val horizontalPadding = animateDpAsState(if (isPressed.read()) 40.dp else 120.dp, tween(1000))

    val style = buttonStyle().let { style ->
      val newColors = style.colors.copy(
        background = background,
        backgroundHovered = background,
        foreground = style.colors.foreground.copy(alpha = alpha),
        foregroundHovered = style.colors.foregroundHovered.copy(alpha = alpha),
      )
      val newDecorators = style.decorations.copy(
        shape = RoundedCornerShape(radius),
        horizontalPadding = horizontalPadding.value,
        verticalPadding = 40.dp
      )
      style.copy(decorations = newDecorators, colors = newColors)
    }

    Row(verticalAlignment = Alignment.CenterVertically) {
      Spacer(Modifier.width(200.dp - horizontalPadding.value)) // :(
      Button(style = style) { isPressed.update { !it } }
    }
  }
}


private val EasingInformation = listOf(
  "LinearEasing" to LinearEasing,
  "FastOutSlowInEasing" to FastOutSlowInEasing,
  "LinearOutSlowInEasing" to LinearOutSlowInEasing,
  "FastOutLinearInEasing" to FastOutLinearInEasing,
  "EaseInSine" to EaseInSine,
  "EaseOutSine" to EaseOutSine,
  "EaseInOutSine" to EaseInOutSine,
  "EaseInQuad" to EaseInQuad,
  "EaseOutQuad" to EaseOutQuad,
  "EaseInOutQuad" to EaseInOutQuad,
  "EaseInCubic" to EaseInCubic,
  "EaseOutCubic" to EaseOutCubic,
  "EaseInOutCubic" to EaseInOutCubic,
  "EaseInQuart" to EaseInQuart,
  "EaseOutQuart" to EaseOutQuart,
  "EaseInOutQuart" to EaseInOutQuart,
  "EaseInQuint" to EaseInQuint,
  "EaseOutQuint" to EaseOutQuint,
  "EaseInOutQuint" to EaseInOutQuint,
  "EaseInExpo" to EaseInExpo,
  "EaseOutExpo" to EaseOutExpo,
  "EaseInOutExpo" to EaseInOutExpo,
  "EaseInCirc" to EaseInCirc,
  "EaseOutCirc" to EaseOutCirc,
  "EaseInOutCirc" to EaseInOutCirc,
  "EaseInBack" to EaseInBack,
  "EaseOutBack" to EaseOutBack
)
