package noria.ui.examples

import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.runtime.*
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.input.pointer.*
import androidx.compose.ui.unit.DpOffset
import androidx.compose.ui.unit.dp
import androidx.compose.ui.util.fastFirstOrNull
import fleet.compose.theme.components.Thickness
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.modifiers.clickable
import fleet.compose.theme.density
import fleet.compose.theme.graphics.css
import fleet.compose.theme.keys.TextStyleKeys
import fleet.compose.theme.keys.ThemeKeys
import fleet.compose.theme.surface
import fleet.compose.theme.text.NoriaParagraphStyle
import fleet.compose.theme.theme
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import noria.NoriaContext
import noria.robot.awt.mouseMove
import noria.robot.awt.withRobot
import noria.ui.components.dropDownList
import noria.ui.text.WebLink
import noria.ui.withModifier
import noria.windowManagement.api.MouseButton
import java.util.*
import kotlin.math.pow
import kotlin.math.sqrt

internal fun vsyncTesterExamples(): Gallery = gallery("VSync Tester", NoriaExamples.sourceCodeForFile("VSyncTester.kt")) {
  example("Vsync Tester") {
    Column {
      UiText("Click to toggle animation. When v-sync works smooth, text should looks grey.",
             paragraphStyle = NoriaParagraphStyle.multiline)
      Spacer(Modifier.height(4.dp))
      WebLink("https://www.vsynctester.com/manual.html")
      Spacer(Modifier.height(4.dp))
      vsyncTester()
      Spacer(Modifier.height(4.dp))
      inputLagDetector()
    }
  }
}

@Composable
fun NoriaContext.vsyncTester() {
  var isAnimating by remember { mutableStateOf(false) }
  var frameIsOdd by remember { mutableStateOf(false) }

  if (isAnimating) {
    LaunchedEffect(Unit) {
      while (true) {
        withFrameMillis {
          frameIsOdd = !frameIsOdd
        }
      }
    }
  }

  val border = if (isAnimating) Color.css("0e6b0e") else Color.Transparent

  withModifier(Modifier
                 .surface(Color(224, 224, 224), 4.dp, border, shape = RoundedCornerShape(size = 8.dp))
                 .padding(all = 4.dp)
                 .clickable(onClick = { isAnimating = !isAnimating })) {
    val color = if (frameIsOdd) {
      Color(255, 128, 128)
    } else {
      Color(128, 255, 255)
    }
    UiText("VSYNC", textStyleKey = TextStyleKeys.Header0SemiBold, color = color)
  }
}


private fun CoroutineScope.moveMouse(initialPosition: DpOffset) {
  launch(Dispatchers.IO) {
    var cursorPosition = initialPosition
    withRobot { robot ->
      repeat(100) {
        robot.mouseMove(to = cursorPosition)
        delay(8)
        cursorPosition += DpOffset(15.dp, 0.dp)
      }
    }
  }
}

@Composable
@OptIn(ExperimentalComposeUiApi::class)
fun NoriaContext.inputLagDetector() {
  var cursorPosition by remember { mutableStateOf<Offset?>(null) }
  var framesAhead by remember { mutableStateOf(1) }
  var pointerIcon by remember { mutableStateOf(PointerIcon.Crosshair) }

  class CursorVelocityTracker {
    val windowSize = 5
    val positions: Queue<Offset> = LinkedList()
    var velocity: Offset? = null

    fun update(position: Offset) {
      if (positions.size >= windowSize) {
        positions.remove()
      }
      positions.add(position)
      val velocities = positions.zipWithNext { prev, next ->
        next - prev
      }
      velocity = velocities.fold(Offset.Zero) { acc, vel ->
        acc + vel
      }.let {
        if (velocities.isNotEmpty()) {
          it.div(velocities.size.toFloat())
        } else {
          it
        }
      }
    }
  }

  val cursorVelocityTracker = remember { CursorVelocityTracker() }

  val maxFramesAhead = 10
  val coroutineScope = rememberCoroutineScope()
  val density = density
  val modifier = Modifier
    .onPointerEvent(PointerEventType.Press) { event ->
      val buttons = event.buttons
      val keyboardModifiers = event.keyboardModifiers

      if (buttons.isPrimaryPressed) {
        val delta = if (keyboardModifiers.isShiftPressed) {
          -1
        }
        else {
          1
        }
        framesAhead = (framesAhead + delta).mod(maxFramesAhead)
      }

    }
    .clickable(button = MouseButton.Right,
               onClick = { with(density) { coroutineScope.moveMouse(it.positionOnScreen.toDpOffset()) } })
    .pointerHoverIcon(pointerIcon)
    .onPointerEvent(PointerEventType.Move) { moveEvent ->
      moveEvent.changes.fastFirstOrNull { it.positionChanged() }?.let { move ->
        cursorPosition = move.position
      }
    }
    .onPointerEvent(PointerEventType.Exit) {
      cursorPosition = null
    }
    .background(theme[ThemeKeys.PopupBackground])
    .border(Thickness.Regular, theme[ThemeKeys.Border])

  Column(Modifier.fillMaxWidth(), propagateMinWidth = true) {
    dropDownList(items = listOf(PointerIcon.Default, PointerIcon.Crosshair, PointerIcon.Hand, PointerIcon.Text),
                 selectedItem = pointerIcon,
                 presentationFn = { it.toString() },
                 onSelect = {
                   pointerIcon = it
                 })
    Spacer(Modifier.height(4.dp))
    Box(modifier.height(200.dp).background(Color.LightGray)) {
      Canvas(modifier = Modifier.fillMaxSize()) {
        cursorPosition?.let { cursorPosition ->
          cursorVelocityTracker.update(cursorPosition)
          cursorVelocityTracker.velocity?.let { velocity ->
            val radius = sqrt(velocity.x.pow(2) +
                              velocity.y.pow(2)) * framesAhead
            drawCircle(color = Color.Black,
                       radius = radius,
                       center = cursorPosition,
                       style = Stroke(2f))
            drawCircle(color = Color.Red, radius = 3f, center = cursorPosition + velocity * framesAhead.toFloat())
          }
          drawCircle(color = Color.Blue, radius = 3f, center = cursorPosition)
        }
      }

      Column(modifier = Modifier.padding(4.dp)) {
        UiText("Move your mouse in this area to test the input lag\n" +
               "The cursor should match with red when moved with constant speed.\n" +
               "Left mouse button / Left mouse button + Shift -> increase/decrease expected input lag\n" +
               "Right mouse button -> simulate mouse movement with robot\n" +
               "Expected input lag is ${framesAhead} frame${if (framesAhead != 1) "s" else ""}",
               paragraphStyle = NoriaParagraphStyle.multiline,
               textStyleKey = TextStyleKeys.DefaultMultiline)
      }
    }
  }
}