package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.text.BasicText
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.*
import androidx.compose.ui.text.font.FontVariation
import androidx.compose.ui.text.style.BaselineShift
import androidx.compose.ui.text.style.LineHeightStyle
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.em
import androidx.compose.ui.unit.sp
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.buildThemedAnnotatedString
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.graphics.applyAlpha
import fleet.compose.theme.graphics.css
import fleet.compose.theme.text.FontSpec
import fleet.compose.theme.text.TextSpec
import fleet.compose.theme.text.fontFamilyForName
import noria.ui.components.innerTextInputStyle
import noria.ui.components.textInput
import noria.ui.components.textInputModel
import org.jetbrains.skia.FontWeight

internal fun textBugsExamples(): Gallery = gallery("TextBugs", NoriaExamples.sourceCodeForFile("TextBugs.kt")) {
  example("Different Font Weights") {
    Column {
      for (fontWeight in 50..1000 step 50) {
        UiText(text = "Hello $fontWeight",
               color = Color.Black,
               backgroundColor = Color.White,
               userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                           weight = fontWeight,
                                                           size = 14.sp)))
        }
    }
  }

  example("Different Font Optical Sizes") {
    Column {
      for (opticalSize in 14..32 step 2) {
        UiText(text = "Some reasonably long line of text. Optical Size: $opticalSize",
               color = Color.Black,
               backgroundColor = Color.White,
               userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                           weight = 480,
                                                           size = 13.sp,
                                                           fontVariations = FontVariation.Settings(FontVariation.Setting("opsz", opticalSize.toFloat())))))
        }
    }
  }


  example("UiText AnnotatedString Doesn't Override LineHeight") {
    Row {
      UiText(text = "Base",
             color = Color.White,
             backgroundColor = Color.css("2D2D2D"),
             userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                         weight = FontWeight.NORMAL,
                                                         size = 10.sp),
                                     lineHeight = 14 / 10f))
      UiText(text = buildThemedAnnotatedString {
        text("Working",
             color = Color.White,
             backgroundColor = Color.css("2D2D2D"),
             userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                         weight = FontWeight.NORMAL,
                                                         size = 10.sp),
                                     lineHeight = 14 / 10f)
        )
      }, userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                     weight = FontWeight.NORMAL,
                                                     size = 10.sp)))
      UiText(text = buildThemedAnnotatedString {
        text("Not working",
             color = Color.White,
             backgroundColor = Color.css("2D2D2D"),
             userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                         weight = FontWeight.NORMAL,
                                                         size = 10.sp),
                                     lineHeight = 14 / 10f)
        )
      }, userTextSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                     weight = FontWeight.NORMAL,
                                                     size = 13.sp)))
    }
  }

  example("BasicText AnnotatedString Not Always Overrides LineHeight") {
    Row {
      val DefaultLineHeightStyle = LineHeightStyle(
        alignment = LineHeightStyle.Alignment.Center,
        trim = LineHeightStyle.Trim.None,
      )

      BasicText(text = "Base", style = TextStyle(color = Color.White,
                                                 background = Color.css("2D2D2D"),
                                                 fontFamily = fontFamilyForName("Inter"),
                                                 fontSize = 10.sp,
                                                 lineHeight = (14 / 10f).em,
                                                 lineHeightStyle = DefaultLineHeightStyle))

      BasicText(text = buildAnnotatedString {
        withStyle(SpanStyle(color = Color.White,
                            background = Color.css("2D2D2D"),
                            fontFamily = fontFamilyForName("Inter"),
                            fontSize = 10.sp)) {
          withStyle(ParagraphStyle(lineHeight = (14 / 10f).em,
                                   lineHeightStyle = DefaultLineHeightStyle)) {
            append("Working")
          }
        }
      },
                style = TextStyle(color = Color.White,
                                  background = Color.css("2D2D2D"),
                                  fontFamily = fontFamilyForName("Inter"),
                                  fontSize = 10.sp,
                                  lineHeightStyle = DefaultLineHeightStyle))

      /*
       * Line height may be determined by font size in style outside of AnnotatedString
       */
      BasicText(text = buildAnnotatedString {
        withStyle(SpanStyle(color = Color.White,
                            background = Color.css("2D2D2D"),
                            fontFamily = fontFamilyForName("Inter"),
                            fontSize = 10.sp)) {
          withStyle(ParagraphStyle(lineHeight = (14 / 10f).em,
                                   lineHeightStyle = DefaultLineHeightStyle)) {
            append("Not Working")
          }
        }
      },
                style = TextStyle(color = Color.White,
                                  background = Color.css("2D2D2D"),
                                  fontFamily = fontFamilyForName("Inter"),
                                  fontSize = 5.sp,
                                  lineHeightStyle = DefaultLineHeightStyle))

    }
  }

  example("BasicText AnnotatedString Overrides LineHeight") {
    Row(verticalAlignment = Alignment.CenterVertically) {
        val lineHeight = 200.sp
        val DefaultLineHeightStyle = LineHeightStyle(
            alignment = LineHeightStyle.Alignment.Center,
            trim = LineHeightStyle.Trim.None,
        )

        BasicText(
          buildAnnotatedString {
                withStyle(ParagraphStyle(lineHeight = lineHeight, lineHeightStyle = DefaultLineHeightStyle)) {
                    append("BasicText LineHeight should be $lineHeight")
                }
            },
          style = TextStyle(lineHeight = 20.sp, lineHeightStyle = DefaultLineHeightStyle),
          modifier = Modifier.background(Color.Cyan)
        )

        Spacer(Modifier.width(20.dp))

        BasicText(
            "BasicText LineHeight should be $lineHeight",
            style = TextStyle(lineHeight = lineHeight, lineHeightStyle = DefaultLineHeightStyle),
            modifier = Modifier.background(Color.Cyan)
        )
    }
  }

  example("Header 2, Does Not Respond To Baseline Shift") {
    Row {
      val textSpec = TextSpec(fontSpec = FontSpec(family = "Inter",
                                                  weight = FontWeight.NORMAL,
                                                  size = 19.sp),
                              lineHeight = 24f / 19f)
      val baselineShiftModel = textInputModel(textSpec.baselineShift.multiplier.toString())
      val baselineShift = baselineShiftModel.state.read().content.toFloatOrNull()?.let {
        BaselineShift(it)
      } ?: BaselineShift.None
      val textSpecWithBaseline = textSpec.copy(baselineShift = baselineShift)
      val annotatedText = buildThemedAnnotatedString {
        text("H2 ", userTextSpec = textSpecWithBaseline, backgroundColor = Color.LightGray.applyAlpha(0.3f))
      }

      UiText(annotatedText, userTextSpec = textSpecWithBaseline)
      UiText("H2", userTextSpec = textSpecWithBaseline, backgroundColor = Color.LightGray.applyAlpha(0.3f))
      Spacer(Modifier.width(8.dp))
      UiText("Baseline Shift:")
      Spacer(Modifier.width(4.dp))
      textInput(baselineShiftModel, style = innerTextInputStyle())
    }
  }
}

