package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.runtime.Composable
import androidx.compose.runtime.State
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.focusTarget
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import fleet.compose.foundation.input.InputEvent
import fleet.compose.foundation.input.inputHandler
import fleet.compose.runtime.CellConsumer
import fleet.compose.runtime.ComposeCell
import fleet.compose.runtime.rememberCell
import fleet.compose.theme.components.Button
import fleet.compose.theme.components.Thickness
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.graphics.applyAlpha
import fleet.compose.theme.keys.ThemeKeys
import fleet.compose.theme.launchOnce
import fleet.compose.theme.launchRestart
import fleet.compose.theme.theme
import kotlinx.coroutines.delay
import noria.*
import noria.model.Propagate
import noria.ui.components.scroll
import noria.ui.components.textInput
import noria.ui.components.withMouseInput
import noria.ui.core.ScrollDirection
import noria.ui.core.boundary
import noria.ui.withModifier
import noria.windowManagement.api.ElementState
import noria.windowManagement.extensions.Keys
import noria.windowManagement.extensions.key
import noria.windowManagement.extensions.state

@Composable
private fun NoriaContext.withCount(count: Int, text: String, builder: @Composable NoriaContext.() -> Unit) {
  withModifier(Modifier.border(Thickness.Regular, theme[ThemeKeys.BackgroundPrimary], shape = RoundedCornerShape(8.dp))) {
    Box(modifier = Modifier.padding(8.dp), propagateMinConstraints = true) {
      Box(contentAlignment = Alignment.Center) {
        builder()
      }

      Box(Modifier.matchParentSize(), propagateMinConstraints = true) {
        Box(contentAlignment = Alignment.BottomStart) {
          UiText("$text: $count", color = theme[ThemeKeys.TextDimmed])
        }
      }
    }
  }
}

@Composable
private fun NoriaContext.blinkOnRerender(builder: @Composable NoriaContext.() -> Unit) {
  val rerender = state { false }
  launchRestart(WILDCARD) {
    delay(200)
    rerender.update { false }
  }

  rerender.update { true }
  Box(propagateMinConstraints = true) {
    builder()

    Box(Modifier.matchParentSize(), propagateMinConstraints = true) {
      boundary {
        if (rerender.read()) {
          Box(
            Modifier
              .background(Color.Green.copy(alpha = .2f))
              .fillMaxSize(),
          )
        }
      }
    }
  }
}

internal fun reactivityExamples(): Gallery = gallery("Reactivity", NoriaExamples.sourceCodeForFile("Reactivity.kt")) {

  /**
   * Boundary allows you to control rerendering of subtrees
   */
  /**
   * Boundary allows you to control rerendering of subtrees
   */
  example("Simple Boundary") {
    val counter = state { 42 }
    Column {
      Row(verticalAlignment = Alignment.CenterVertically) {
        Button(text = "Increment") {
          counter.update { it + 1 }
        }
        Spacer(Modifier.width(8.dp))
        UiText("value: ${counter.read()}")
      }
      Spacer(Modifier.width(8.dp))
      blinkOnRerender {
        UiText("I will blink when counter updates", modifier = Modifier.padding(vertical = 8.dp))
      }
      boundary {
        blinkOnRerender {
          UiText("And I won't", modifier = Modifier.padding(vertical = 8.dp))
        }
      }
    }
  }

  /**
   * This example shows how to use "expr { }" to build incremental computation trees.
   * First column consists of editable inputCells, each one of them returns a thunk.
   * Second and third columns consist of sumCelss, which receive two thunks as input
   * and return a thunk corresponding to its sum. SumCell also shows how many times it
   * was reevaluated. As you can see thunk reevaluates only when its dependencies changes
   */

  /**
   * This example shows how to use "expr { }" to build incremental computation trees.
   * First column consists of editable inputCells, each one of them returns a thunk.
   * Second and third columns consist of sumCelss, which receive two thunks as input
   * and return a thunk corresponding to its sum. SumCell also shows how many times it
   * was reevaluated. As you can see thunk reevaluates only when its dependencies changes
   */
  example("Incremental Cells", preferredHeight = 600.dp) {

    val cellWidth = 80.dp
    val cellHeight = cellWidth

    @Composable
    fun NoriaContext.inputState(initialValue: Int): State<Int?> {
      val valueState = remember { mutableStateOf(initialValue) }
      Row(
        Modifier
          .size(cellWidth, cellHeight)
          .padding(horizontal = 8.dp, vertical = 8.dp)
          .border(Thickness.Regular, theme[ThemeKeys.TextPrimary], shape = RoundedCornerShape(8.dp)),
        verticalAlignment = Alignment.CenterVertically,
      ) {
        textInput(
          valueState.value.toString() ?: "",
          Modifier.weight(1f),
          onInput = { newContent ->
            valueState.value = newContent.toIntOrNull() ?: valueState.value
          })
      }
      return valueState
    }

    @Composable
    fun NoriaContext.sumCell(expr1: State<Int?>, expr2: State<Int?>): ComposeCell<Int?> {
      val reevalCount = state { 0 }
      val sumThunk = rememberCell {
        reevalCount.update { i -> i + 1 }
        val val1 = expr1.value
        val val2 = expr2.value

        if (val1 != null && val2 != null) {
          val1 + val2
        }
        else {
          null
        }
      }

      sideEffect {
        reevalCount.update { 0 }
      }

      withModifier(Modifier.padding(horizontal = 8.dp, vertical = 8.dp)) {
        withModifier(Modifier.size(width = cellWidth,
                                   height = cellHeight)) {
          withModifier(Modifier.border(Thickness.Regular, theme[ThemeKeys.TextPrimary], shape = RoundedCornerShape(8.dp))) {
            withCount(reevalCount.read(), "evals") {
              CellConsumer(sumThunk) { sumState ->
                UiText(sumState.value.toString())
              }
            }
          }
        }
      }

      return sumThunk
    }

    var e1: State<Int?>? = null
    var e2: State<Int?>? = null
    var e3: State<Int?>? = null
    var e4: State<Int?>? = null

    var s1: ComposeCell<Int?>? = null
    var s2: ComposeCell<Int?>? = null

    Row(verticalAlignment = Alignment.CenterVertically) {
      Column {
        e1 = inputState(4)
        e2 = inputState(8)
        Spacer(Modifier.width(16.dp))
        e3 = inputState(15)
        e4 = inputState(16)
      }
      Column {
        Row(verticalAlignment = Alignment.CenterVertically) {
          UiText("sum=")
          s1 = sumCell(e1!!, e2!!)
        }
        Spacer(Modifier.height(cellHeight + 16.dp))
        Row(verticalAlignment = Alignment.CenterVertically) {
          UiText("sum=")
          s2 = sumCell(e3!!, e4!!)
        }
      }
      Column {
        Row(verticalAlignment = Alignment.CenterVertically) {
          UiText("sum=")
          CellConsumer(s1!!, s2!!) { s1State, s2State ->
            sumCell(s1State, s2State)
          }
        }
      }
    }
  }

  /**
   * Example of how to use the keyboard to move a box from left to right
   */

  /**
   * Example of how to use the keyboard to move a box from left to right
   */
  example("Moving Box with Keyboard", preferredHeight = 600.dp) {
    val speed = state { 0 }

    val padding = state { 0 }
    launchOnce {
      while (true) {
        padding.update {
          it + speed.readNonReactive()
        }
        delay(16)
      }
    }
    val focusRequester = remember { FocusRequester() }
    Box(modifier = Modifier.focusRequester(focusRequester).focusTarget().inputHandler { e ->
      if (e is InputEvent.RawKeyboardInput) {
        val keyboardInputEvent = e.keyboardInputEvent
        val key = keyboardInputEvent.key
        when (keyboardInputEvent.state) {
          ElementState.Pressed -> {
            speed.update { currentValue ->
              when (key) {
                Keys.Right -> 1
                Keys.Left -> -1
                else -> {
                  currentValue
                }
              }
            }
          }
          ElementState.Released -> {
            speed.update { currentValue ->
              when (key) {
                Keys.Right -> 0
                Keys.Left -> 0
                else -> {
                  currentValue
                }
              }
            }
          }
        }

        if (key == Keys.Left || key == Keys.Right) Propagate.STOP else Propagate.CONTINUE
      }
      else Propagate.CONTINUE
    }) {
      withMouseInput(onMouseDown = { focusRequester.requestFocus() }) {
        Column(modifier = Modifier
          .border(width = 2.dp, Color.Green)
          .padding(20.dp)
        ) {
          UiText("Use left and right keys to move the box")
          scroll(direction = ScrollDirection.BOTH) {
            Box(
              Modifier
                .padding(start = padding.read().dp)
                .background(Color.Red.applyAlpha(.5f))
                .size(16.dp)
            )
          }
        }
      }
    }
  }
}
