package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.modifiers.clickable
import fleet.compose.theme.overlay.overlay
import noria.NoriaContext
import noria.state
import noria.ui.components.*
import noria.ui.core.MainOverlayHost
import noria.ui.withModifier

internal fun popupExamples(): Gallery = gallery("Popup", NoriaExamples.sourceCodeForFile("Popup.kt")) {
  example("Custom Modifier Positioning") {
    @Composable
    fun NoriaContext.PositionedPopup(place: PopupPlace, direction: PopupDirection) {
      val placeVertical = place.toVerticalAlignment()
      val placeHorizontal = place.toHorizontalAlignment()
      val directionVertical = direction.toVerticalAlignment()
      val directionHorizontal = direction.toHorizontalAlignment()

      Box(modifier = Modifier.overlay(MainOverlayHost) {
        val alignModifier = if (placeVertical != null && directionHorizontal != null) {
          Modifier.alignByAnchor(placeVertical, directionHorizontal)
        }
        else if (placeHorizontal != null && directionVertical != null) {
          Modifier.alignByAnchor(placeHorizontal, directionVertical)
        }
        else {
          error("unsupported popup placement $place, $direction")
        }
        UiText(
          text = "I am Popup",
          color = Color.White,
          modifier = Modifier
            .then(alignModifier)
            .background(Color.Blue)
            .padding(vertical = 5.dp, horizontal = 2.dp)
        )
      }) {
        UiText(
          text = "Popup Anchor",
          color = Color.White,
          modifier = Modifier
            .background(Color.Red)
            .padding(vertical = 30.dp, horizontal = 15.dp)
        )
      }
    }

    Column {
      Spacer(Modifier.height(30.dp))
      val position = state { PopupPlace.Left }
      val direction = if (position.read() == PopupPlace.Bottom || position.read() == PopupPlace.Top) {
        state { PopupDirection.LeftToRight }
      }
      else {
        state { PopupDirection.TopToBottom }
      }
      Row {
        dropDownList(PopupPlace.entries, position) { it.name }
        Spacer(Modifier.width(30.dp))
        if (position.read() == PopupPlace.Bottom || position.read() == PopupPlace.Top) {
          dropDownList(listOf(PopupDirection.LeftToRight, PopupDirection.RightToLeft),
                       direction) { it.name }
        }
        else {
          dropDownList(listOf(PopupDirection.TopToBottom, PopupDirection.BottomToTop),
                       direction) { it.name }
        }
      }
      withModifier(Modifier.fillMaxWidth()) {
        Box(contentAlignment = Alignment.Center) {
          withModifier(Modifier.padding(vertical = 80.dp)) {
            PositionedPopup(
              place = position.read(),
              direction = direction.read()
            )
          }
        }
      }
    }
  }
  example("Default Positioning") {
    withPopup(
      popupBuilder = {
        UiText(text = "I am Popup",
               color = Color.White,
               modifier = Modifier
                 .background(Color.Blue)
                 .padding(horizontal = 2.dp, vertical = 5.dp))
      },
    ) {
      UiText(text = "Popup Anchor",
             color = Color.White,
             modifier = Modifier
               .background(Color.Red)
               .padding(vertical = 30.dp, horizontal = 15.dp))
    }
  }

  /**
   * Position popups using AnchorJoint and PopupJoint
   */

  /**
   * Position popups using AnchorJoint and PopupJoint
   */

  /**
   * Position popups using AnchorJoint and PopupJoint
   */

  /**
   * Position popups using AnchorJoint and PopupJoint
   */
  example("Custom Positioning") {
    @Composable
    fun NoriaContext.positionedPopup(place: PopupPlace, direction: PopupDirection) {
      withPopup(
        popupBuilder = {
          UiText(text = "I am Popup",
                 color = Color.White,
                 modifier = Modifier
                   .background(Color.Blue)
                   .padding(vertical = 5.dp, horizontal = 2.dp))
        },
        position = PopupPosition(place = place, direction = direction),
      ) {
        UiText(text = "Popup Anchor",
               color = Color.White,
               modifier = Modifier
                 .background(Color.Red)
                 .padding(vertical = 30.dp, horizontal = 15.dp))
      }
    }

    Column {
      Spacer(Modifier.height(30.dp))
      val position = state { PopupPlace.Left }
      val direction = if (position.read() == PopupPlace.Bottom || position.read() == PopupPlace.Top) {
        state { PopupDirection.LeftToRight }
      }
      else {
        state { PopupDirection.TopToBottom }
      }
      Row {
        dropDownList(PopupPlace.entries, position) { it.name }
        Spacer(Modifier.width(30.dp))
        if (position.read() == PopupPlace.Bottom || position.read() == PopupPlace.Top) {
          dropDownList(listOf(PopupDirection.LeftToRight, PopupDirection.RightToLeft),
                       direction) { it.name }
        }
        else {
          dropDownList(listOf(PopupDirection.TopToBottom, PopupDirection.BottomToTop),
                       direction) { it.name }
        }
      }
      withModifier(Modifier.fillMaxWidth()) {
        Box(contentAlignment = Alignment.Center) {
          withModifier(Modifier.padding(vertical = 80.dp)) {
            positionedPopup(
              place = position.read(),
              direction = direction.read()
            )
          }
        }
      }
    }
  }

  example("Hiding Popup") {
    val popupIsShown = state { false }
    withCloseablePopup(
      shownState = popupIsShown,
      disableClickOnAnchor = false,
      popupBuilder = {
        UiText(text = "I am a Popup",
               color = Color.White,
               modifier = Modifier
                 .background(Color.Blue)
                 .padding(vertical = 5.dp, horizontal = 2.dp))
      },
    ) {
      UiText(
        text = "Click Me!",
        color = Color.White,
        modifier = Modifier
          .clickable(onClick = {
            popupIsShown.update { !it }
          })
          .background(Color.Red)
          .padding(vertical = 30.dp, horizontal = 15.dp)
      )
    }
  }

  example("Popup Button") {
    withModifier(Modifier.fillMaxWidth()) {
      Box(contentAlignment = Alignment.Center) {
        withModifier(Modifier.padding(vertical = 16.dp)) {
          popupButton(
            popupBuilder = {
              UiText(text = "I am a Popup",
                     color = Color.White,
                     modifier = Modifier
                       .background(Color.Blue)
                       .padding(vertical = 1.dp, horizontal = 2.dp))
            },
          ) { colors, _ ->
            UiText("Click Me!", colors.foreground, modifier = Modifier.padding(4.dp))
          }
        }
      }
    }
  }
}

private fun PopupPlace.toHorizontalAlignment(): Alignment.Horizontal? = when (this) {
  PopupPlace.Left -> Alignment.Start
  PopupPlace.Top -> null
  PopupPlace.Right -> Alignment.End
  PopupPlace.Bottom -> null
}

private fun PopupPlace.toVerticalAlignment(): Alignment.Vertical? = when (this) {
  PopupPlace.Left -> null
  PopupPlace.Top -> Alignment.Top
  PopupPlace.Right -> null
  PopupPlace.Bottom -> Alignment.Bottom
}


private fun PopupDirection.toVerticalAlignment(): Alignment.Vertical? = when (this) {
  PopupDirection.LeftToRight -> null
  PopupDirection.RightToLeft -> null
  PopupDirection.TopToBottom -> Alignment.Top
  PopupDirection.BottomToTop -> Alignment.Bottom
}

private fun PopupDirection.toHorizontalAlignment(): Alignment.Horizontal? = when (this) {
  PopupDirection.LeftToRight -> Alignment.Start
  PopupDirection.RightToLeft -> Alignment.End
  PopupDirection.TopToBottom -> null
  PopupDirection.BottomToTop -> null
}