package noria.ui.examples

import androidx.compose.foundation.hoverable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.interaction.noriaCollectIsHoveredAsState
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import fleet.compose.theme.components.Button
import fleet.util.AtomicRef
import fleet.util.getAndIncrement
import kotlinx.collections.immutable.persistentListOf
import noria.*
import noria.ui.components.NotebookCellState
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.horizontalSeparator
import noria.ui.components.notebook
import noria.ui.components.scroll
import noria.ui.core.ScrollDirection
import fleet.compose.theme.components.UiText
import noria.ui.withModifier
import kotlin.math.abs
import kotlin.random.Random

internal fun notebookExample(): Gallery = gallery("Notebook", NoriaExamples.sourceCodeForFile("Notebook.kt")) {
  example("Raw State") {
    val cells = state { listOf(newCell(), newCell()) }
    val updates = state { persistentListOf<String>() }
    Row {
      withModifier(Modifier.weight(2f)) {
        notebook(
          cellStates = cells.read(),
          notebookWrapper = { cellList ->
            updates.update { it.add("Render notebook") }
            Button(text = "Update Random Cell") {
              cells.update {
                val id = abs(Random.nextInt()) % it.size
                it.take(id) + it[id].copy(state = randText()) + it.drop(id + 1)
              }
            }
            Spacer(Modifier.height(8.dp))
            cellList()
          },
          cellRenderFn = { id, state ->
            updates.update { it.add("Render cell $id") }
            UiText("$id: $state")
          },
          separator = { idx ->
            val interactionSource = remember { MutableInteractionSource() }
            val isHovered by interactionSource.noriaCollectIsHoveredAsState(this)
            Box(Modifier.fillMaxWidth().hoverable(interactionSource), propagateMinConstraints = true) {
              if (isHovered) {
                Box(contentAlignment = Alignment.Center) {
                  Button(text = "Add Cell") {
                    cells.update {
                      it.take(idx + 1) + newCell() + it.drop(idx + 1)
                    }
                  }
                }
              }
              else {
                horizontalSeparator(Modifier.padding(vertical = 4.dp))
              }
            }
          }
        )
      }
      withModifier(Modifier.weight(1f)) {
        scroll(ScrollDirection.VERTICAL) {
          Column {
            for (update in updates.read()) {
              UiText(update)
            }
          }
        }
      }
    }
  }

  example("State Cell") {
    val updateRequests = state { persistentListOf(0, 1) }
    val cells = state { emptyList<NotebookCellState<Int, StateCell<String>>>() }
    val updates = state { persistentListOf<String>() }
    Row {
      withModifier(Modifier.weight(2f)) {
        val newCells = updateRequests.read().map { it to newStateCell() }
        updateRequests.update { persistentListOf() }
        val actualCells = cells.read().add(newCells)
        cells.update { actualCells }
        notebook(
          cellStates = cells.read(),
          notebookWrapper = { cellList ->
            updates.update { it.add("Render notebook") }
            Button(text = "Update Random Cell") {
              val id = abs(Random.nextInt()) % cells.readNonReactive().size
              cells.readNonReactive()[id].state.update { randText() }
            }
            Spacer(Modifier.height(8.dp))
            cellList()
          },
          cellRenderFn = { id, state ->
            updates.update { it.add("Render cell $id") }
            UiText("$id: ${state.read()}")
          },
          separator = { idx ->
            val interactionSource = remember { MutableInteractionSource() }
            val isHovered by interactionSource.noriaCollectIsHoveredAsState(this)
            Box(Modifier.fillMaxWidth().hoverable(interactionSource), propagateMinConstraints = true) {
              if (isHovered) {
                Box(contentAlignment = Alignment.Center) {
                  Button(text = "Add Cell") {
                    updateRequests.update { it.add(idx + 1) }
                  }
                }
              }
              else {
                horizontalSeparator(Modifier.padding(vertical = 4.dp))
              }
            }
          }
        )
      }
      withModifier(Modifier.weight(1f)) {
        scroll(ScrollDirection.VERTICAL) {
          Column {
            for (update in updates.read()) {
              UiText(update)
            }
          }
        }
      }
    }
  }
}

private fun List<NotebookCellState<Int, StateCell<String>>>.add(
  new: List<Pair<Int, NotebookCellState<Int, StateCell<String>>>>
): List<NotebookCellState<Int, StateCell<String>>> {
  val res = toMutableList()
  for ((idx, value) in new) {
    res.add(idx, value)
  }
  return res
}

private fun randText(): String = Random.nextInt(0, 100).toString()

private fun newCell(): NotebookCellState<Int, String> = NotebookCellState(counter.getAndIncrement(), randText())

@Composable
private fun NoriaContext.newStateCell(): NotebookCellState<Int, StateCell<String>> {
  val id = stateCounter.getAndIncrement()
  val state = scope(id) { state { randText() } }
  return NotebookCellState(id, state)
}

private var counter = AtomicRef(0)
private var stateCounter = AtomicRef(0)
