package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.RectangleShape
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.IntSize
import androidx.compose.ui.unit.constrain
import androidx.compose.ui.unit.dp
import fleet.compose.theme.Edge
import fleet.compose.theme.components.Thickness
import fleet.compose.theme.graphics.applyAlpha
import fleet.compose.theme.layout.ProposedSize
import fleet.compose.theme.layout.Resizable
import fleet.compose.theme.surface
import fleet.util.enumSetOf
import noria.NoriaContext
import noria.state
import noria.ui.components.*
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import noria.ui.core.layout
import fleet.compose.theme.theme
import fleet.compose.theme.components.UiText
import fleet.compose.theme.keys.ThemeKeys
import noria.ui.withModifier

internal fun layoutExamples(): Gallery = gallery("Layout", NoriaExamples.sourceCodeForFile("Layout.kt")) {
  /**
   * Constraints define the ranges for width and height that a component may choose from during layout.
   *
   * If the minimum and maximum size are the same, then the component must lay itself out at the given size, even if it would like to use
   * a different size. This situation is called "tight constraints".
   *
   * This box is stretched horizontally even though it would like to only take up 10 units of width.
   */
  /**
   * Constraints define the ranges for width and height that a component may choose from during layout.
   *
   * If the minimum and maximum size are the same, then the component must lay itself out at the given size, even if it would like to use
   * a different size. This situation is called "tight constraints".
   *
   * This box is stretched horizontally even though it would like to only take up 10 units of width.
   */
  example("Tight Constraints") {
    Box(
      Modifier
        .background(Color.Green.applyAlpha(.5f))
        .size(40.dp)
    )
    // The explicit size only applies to the height. It has no effect on the width due to tight constraints
  }

  example("Loosening Constraints (Single Child)") {
    Column(Modifier.fillMaxWidth(), propagateMinWidth = true) {
      UiText("align")
      Spacer(Modifier.height(4.dp))
      withModifier(Modifier.height(height = 80.dp)) {
        withModifier(Modifier.border(Thickness.Regular, Color.Gray)) {
          Box(contentAlignment = Alignment.TopStart) {
            Box(
              Modifier
                .background(Color.Green.applyAlpha(.5f))
                .size(40.dp)
            )
          }
        }
      }
      Spacer(Modifier.height(8.dp))

      UiText("alignToTheRight")
      Spacer(Modifier.height(4.dp))
      withModifier(Modifier.height(height = 80.dp)) {
        withModifier(Modifier.border(Thickness.Regular, Color.Gray)) {
          Box(contentAlignment = Alignment.TopEnd) {
            Box(
              Modifier
                .background(Color.Green.applyAlpha(.5f))
                .size(40.dp)
            )
          }
        }
      }
      Spacer(Modifier.height(8.dp))

      UiText("alignToTheBottom")
      Spacer(Modifier.height(4.dp))
      withModifier(Modifier.height(height = 80.dp)) {
        withModifier(Modifier.border(Thickness.Regular, Color.Gray)) {
          Box(contentAlignment = Alignment.BottomStart) {
            Box(
              Modifier
                .background(Color.Green.applyAlpha(.5f))
                .size(40.dp)
            )
          }
        }
      }
      Spacer(Modifier.height(8.dp))

      UiText("centerHorizontally")
      Spacer(Modifier.height(4.dp))
      withModifier(Modifier.height(height = 80.dp)) {
        withModifier(Modifier.border(Thickness.Regular, Color.Gray)) {
          Box(contentAlignment = Alignment.Center) {
            Box(
              Modifier
                .background(Color.Green.applyAlpha(.5f))
                .size(40.dp)
            )
          }
        }
      }
      Spacer(Modifier.height(8.dp))

      Spacer(Modifier.height(8.dp))
      UiText("centerVertically")
      Spacer(Modifier.height(4.dp))
      withModifier(Modifier.height(height = 80.dp)) {
        withModifier(Modifier.border(Thickness.Regular, Color.Gray)) {
          Box(contentAlignment = Alignment.Center) {
            Box(
              Modifier
                .background(Color.Green.applyAlpha(.5f))
                .size(40.dp)
            )
          }
        }
      }
      Spacer(Modifier.height(8.dp))

      Spacer(Modifier.height(8.dp))
      UiText("center")
      Spacer(Modifier.height(4.dp))
      withModifier(Modifier.height(height = 80.dp)) {
        withModifier(Modifier.border(Thickness.Regular, Color.Gray)) {
          Box(contentAlignment = Alignment.Center) {
            Box(
              Modifier
                .background(Color.Green.applyAlpha(.5f))
                .size(40.dp)
            )
          }
        }
      }
      Spacer(Modifier.height(8.dp))

      UiText("align with custom arguments")
      Spacer(Modifier.height(8.dp))
    }
  }

  example("Stack") {
    Box(contentAlignment = Alignment.TopStart) {
      Box(propagateMinConstraints = true) {
        Box(
          Modifier
            .background(Color.Red.applyAlpha(.5f))
            .size(size = 56.dp)
        )

        Box(Modifier.matchParentSize(), propagateMinConstraints = true) {
          UiText("Some Content")

          Box(
            Modifier
              .padding(start = 16.dp, top = 16.dp)
              .background(Color.Green.applyAlpha(.5f))
              .size(40.dp)
          )

          Box(
            Modifier
              .padding(start = 32.dp, top = 32.dp)
              .background(Color.Blue.applyAlpha(.5f))
              .size(40.dp)
          )
        }
      }
    }
  }

  /**
   * Resize the box to change the total width
   */

  /**
   * Resize the box to change the total width
   */
  example("Growing & Shrinking") {
    val totalSizeState = state { ProposedSize(400.dp, 40.dp) }
    Column {
      UiText("Total width: ${totalSizeState.read().width!! / 4.dp}")
      withModifier(Modifier
                     .background(theme[ThemeKeys.PopupBackground])
                     .border(Thickness.Regular, theme[ThemeKeys.Border])) {
        Resizable(size = totalSizeState, borders = enumSetOf<Edge>(Edge.End)) {
          Row {
            Box(
              Modifier
                .weight(1f)
                .height(40.dp)
                .background(Color.Red.applyAlpha(0.5f)),
            )
            Box(
              Modifier
                .weight(10f)
                .height(40.dp)
                .background(Color.Green.applyAlpha(0.5f)),
            )
          }
        }
      }
    }
  }

  /**
   * You can use `weight` modifier with `fill = false` to make the component shrink if needed, but never grow.
   *
   * Resize the box to change the total width
   */

  /**
   * You can use `weight` modifier with `fill = false` to make the component shrink if needed, but never grow.
   *
   * Resize the box to change the total width
   */
  example("Shrinking Only") {
    val totalSizeState = state { ProposedSize(400.dp, 40.dp) }
    Column {
      UiText("Total width: ${totalSizeState.read().width!! / 4.dp}")
      withModifier(Modifier.surface(
        container = theme[ThemeKeys.PopupBackground],
        width = Thickness.Regular,
        border = theme[ThemeKeys.Border],
        shape = RectangleShape
      )) {
        Resizable(size = totalSizeState, borders = enumSetOf(Edge.End)) {
          Row {
            Spacer(
              Modifier
                .size(400.dp, 40.dp)
                .weight(1f, fill = false)
                .background(Color.Red.applyAlpha(0.5f))
                .border(3.dp, Color.Red)
            )
          }
        }
      }
    }
  }

  // todo[unterhofer] What is this supposed to demonstrate? I don't see an order in here
  example("Shrinking Order") {
    val totalSizeState = state { ProposedSize(400.dp, 40.dp) }
    Column {
      UiText("Total width: ${totalSizeState.read().width!! / 4.dp}")

      withModifier(
        Modifier
          .background(theme[ThemeKeys.PopupBackground])
          .border(Thickness.Regular, theme[ThemeKeys.Border])
      ) {
        Resizable(size = totalSizeState, borders = enumSetOf<Edge>(Edge.End)) {
          Row {
            Box(
              Modifier
                .size(120.dp, 40.dp)
                .background(Color.Red.applyAlpha(0.5f)),
            )

            Box(
              Modifier
                .weight(1f)
                .height(40.dp)
                .background(Color.Blue.applyAlpha(0.5f)),
            )

            Box(
              Modifier
                .size(120.dp, 40.dp)
                .background(Color.Green.applyAlpha(.5f)),
            )
          }
        }
      }
    }
  }
}
