package noria.ui.examples

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.text.BasicText
import androidx.compose.foundation.text.ClickableText
import androidx.compose.foundation.text.InlineTextContent
import androidx.compose.foundation.text.appendInlineContent
import androidx.compose.foundation.text.selection.DisableSelection
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorProducer
import androidx.compose.ui.graphics.Shadow
import androidx.compose.ui.layout.AlignmentLine
import androidx.compose.ui.layout.FirstBaseline
import androidx.compose.ui.layout.LastBaseline
import androidx.compose.ui.layout.layout
import androidx.compose.ui.text.*
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.platform.Font
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.style.TextGeometricTransform
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.em
import androidx.compose.ui.unit.sp
import fleet.compose.foundation.text.selection.NoriaSelectionContainer
import fleet.compose.theme.LocalFontRasterizationSettings
import fleet.compose.theme.LocalResourceReader
import fleet.compose.theme.LocalTheme
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.buildThemedAnnotatedString
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.components.link
import fleet.compose.theme.keys.TextStyleKeys
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.flow
import noria.NoriaContext
import noria.ui.loop.windowManager
import fleet.compose.theme.text.NoriaParagraphStyle
import noria.windowManagement.extensions.openUrl

@OptIn(ExperimentalFoundationApi::class)
fun composeTextExamples(): Gallery = gallery("Compose Text", NoriaExamples.sourceCodeForFile("ComposeText.kt")) {
  example("BasicText") {
    Row {
      NoriaSelectionContainer {
        Column {
          BasicText("Hello from compose")
          BasicText("Hello from compose", style = TextStyle(
            color = Color.Red,
            fontSize = 16.sp,
            fontFamily = FontFamily.Monospace,
            fontWeight = FontWeight.W800,
            fontStyle = FontStyle.Italic,
            letterSpacing = 0.5.em,
            background = Color.LightGray,
            textDecoration = TextDecoration.Underline
          ), color = ColorProducer { Color.Cyan })
        }
      }
    }
  }
  example("Overflow") {
    val text = "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida massa laoreet ultrices porttitor."
    val bigTextStyle = TextStyle(fontSize = 24.sp)
    val smallTextStyle = TextStyle(fontSize = 18.sp)
    Row {
      NoriaSelectionContainer {
        Box {
          Column(Modifier.width(400.dp).padding(8.dp)) {
            BasicText(text = "Clip", style = bigTextStyle)
            BasicText(
              text = text,
              overflow = TextOverflow.Clip,
              maxLines = 2,
              style = smallTextStyle
            )
            Spacer(modifier = Modifier.height(8.dp))

            BasicText(text = "Ellipsis", style = bigTextStyle)
            BasicText(
              text = text,
              overflow = TextOverflow.Ellipsis,
              maxLines = 2,
              style = smallTextStyle
            )
            Spacer(modifier = Modifier.height(8.dp))

            BasicText(text = "Visible", style = bigTextStyle)
            BasicText(
              text = text,
              overflow = TextOverflow.Visible,
              softWrap = false,
              maxLines = 2,
              style = smallTextStyle
            )
          }
        }
      }
    }
  }

  example("Style") {
    BasicText(
      text = "Jetpack Compose",
      style = TextStyle(
        color = Color.Green,
        fontSize = 24.sp,
        fontFamily = FontFamily.Monospace,
        letterSpacing = 4.sp,
        textAlign = TextAlign.Center,
        shadow = Shadow(
          color = Color.Black,
          offset = Offset(8f, 8f),
          blurRadius = 4f
        ),
        // also doesn't work on CfD: https://github.com/JetBrains/compose-multiplatform/issues/3990
        textGeometricTransform = TextGeometricTransform(
          scaleX = 2.5f,
          skewX = 1f
        )
      ),
      modifier = Modifier.width(300.dp)
    )
  }

  example("AnnotatedString1") {
    BasicText(
      text = buildAnnotatedString {
        append(
          AnnotatedString("AnnotatedString", spanStyle = SpanStyle(Color.Red))
        )
        append('≠')
        append("String")
      },
      style = TextStyle(fontSize = 24.sp) // todo change font familly
    )
  }

  example("AnnotatedString2") {
    BasicText(
      text = buildAnnotatedString {
        append("Hello, ")

        pushStyle(style = SpanStyle(color = Color.Green))
        append("this ")
        append("is ")
        append("example ")
        append("of ")
        append("pushStyle ")
        pop()

        pushStyle(style = SpanStyle(color = Color.Red))
        append("and ")
        append("pop ")
        pop()

        append("methods")
      },
      style = TextStyle(fontSize = 24.sp)
    )
  }

  example("AnnotatedString3") {
    BasicText(
      text = buildAnnotatedString {
        append("Jetpack Compose")
        addStyle(
          style = SpanStyle(
            color = Color.Red,
            fontWeight = FontWeight.Bold
          ),
          start = 0,
          end = 3
        )
        addStyle(
          style = ParagraphStyle(
            textAlign = TextAlign.End
          ),
          start = 8,
          end = 15
        )
        addStyle(
          style = SpanStyle(
            color = Color.Green,
            textDecoration = TextDecoration.Underline
          ),
          start = 8,
          end = 15
        )
      },
      style = TextStyle(fontSize = 24.sp),
      modifier = Modifier.width(300.dp)
    )
  }

  example("AdvancedStyling") {
    val LightBlue = Color(0xFF0066FF)
    val Purple = Color(0xFF800080)
    val gradientColors = listOf(Color.Cyan, LightBlue, Purple /*...*/)

    BasicText(
      text = "Do not allow people to dim your shine because they are blinded. Tell them to put on some sunglasses",
      style = TextStyle(
        fontSize = 24.sp,
        brush = Brush.linearGradient(
          colors = gradientColors
        )
      )
    )
  }

  example("Fonts") {
    val resourceReader = LocalResourceReader.current
    val lobster by flow {
      val lobsterBytes = resourceReader.read("gallery/LobsterTwo-Regular.ttf")!!
      val lobster = FontFamily(Font("LobsterTwo-Regular", lobsterBytes))
      emit(lobster)
    }.noriaCollectAsState(this, FontFamily.Default)

    Column {
      BasicText("Hello World", style = TextStyle(fontFamily = FontFamily.Serif))
      BasicText("Hello World", style = TextStyle(fontFamily = FontFamily.SansSerif))
      BasicText("Hello World", style = TextStyle(fontFamily = lobster))
    }
  }

  example("InlineContent") {
    val myId = "inlineContent"
    val text = buildAnnotatedString {
      append("Hello")
      // Append a placeholder string "[myBox]" and attach an annotation "inlineContent" on it.
      appendInlineContent(myId, "[myBox]")
      append("World")
    }

    var state by remember { mutableStateOf(false) }
    val inlineContent = mapOf(
      Pair(
        // This tells the [BasicText] to replace the placeholder string "[myBox]" by
        // the composable given in the [InlineTextContent] object.
        myId,
        InlineTextContent(
          // Placeholder tells text layout the expected size and vertical alignment of
          // children composable.
          Placeholder(
            width = 2.em,
            height = 2.em,
            placeholderVerticalAlign = PlaceholderVerticalAlign.AboveBaseline
          )
        ) {
          // This [Box] will fill maximum size, which is specified by the [Placeholder]
          // above. Notice the width and height in [Placeholder] are specified in TextUnit,
          // and are converted into pixel by text layout.
          Checkbox(state, onCheckedChange = { state = it })
          //          Box(modifier = Modifier.fillMaxSize().background(color = Color.Red))
        }
      )
    )

    BasicText(text = text, inlineContent = inlineContent)
  }

  example("Links") {
    val annotatedText = buildAnnotatedString {
      append("Click ")

      // We attach this *URL* annotation to the following content
      // until `pop()` is called
      pushStringAnnotation(
        tag = "URL", annotation = "https://jetbrains.com"
      )
      withStyle(
        style = SpanStyle(
          color = Color.Blue, fontWeight = FontWeight.Bold
        )
      ) {
        append("here")
      }

      pop()
    }

    val windowManager = windowManager
    ClickableText(text = annotatedText, onClick = { offset ->
      // We check if there is an *URL* annotation attached to the text
      // at the clicked position
      annotatedText.getStringAnnotations(
        tag = "URL", start = offset, end = offset
      ).firstOrNull()?.let { annotation ->
        windowManager.openUrl(annotation.item)
      }
    })

  }

  example("Emoji") {
    Row {
      NoriaSelectionContainer {
        BasicText(text = "Hello 🫠🫱🏼‍🫲🏿🫰🏽")
      }
    }
  }

  fun Modifier.firstBaselineToTop(
    firstBaselineToTop: Dp,
  ) = layout { measurable, constraints ->
    // Measure the composable
    val placeable = measurable.measure(constraints)

    // Check the composable has a first baseline
    check(placeable[FirstBaseline] != AlignmentLine.Unspecified)
    val firstBaseline = placeable[FirstBaseline]

    // Height of the composable with padding - first baseline
    val placeableY = firstBaselineToTop.roundToPx() - firstBaseline
    val height = placeable.height + placeableY
    layout(placeable.width, height) {
      // Where the composable gets placed
      placeable.placeRelative(0, placeableY)
    }
  }

  example("Baseline1") {
    Row {
      NoriaSelectionContainer {
        Row {
          BasicText("Default position")
          BasicText("With modifier", modifier = Modifier.firstBaselineToTop(32.dp))
        }
      }
    }
  }

  example("Baseline2") {
    Row {
      Column {
        Row(Modifier.background(Color.Red)) {
          BasicText(
            text = "Line 1\nLine 2",
            style = TextStyle(fontSize = 20.sp),
          )

          BasicText(
            text = "Two",
            style = TextStyle(fontSize = 12.sp),
          )
        }
        Row(Modifier.background(Color.Green)) {
          BasicText(
            text = "Line 1\nLine 2",
            style = TextStyle(fontSize = 20.sp),
            modifier = Modifier.alignByBaseline()
          )

          BasicText(
            text = "Two",
            style = TextStyle(fontSize = 12.sp),
            modifier = Modifier.alignByBaseline()
          )
        }
        Row(Modifier.background(Color.Magenta)) {
          BasicText(
            text = "Line 1\nLine 2",
            style = TextStyle(fontSize = 20.sp),
            modifier = Modifier.alignBy(LastBaseline)
          )

          BasicText(
            text = "Two",
            style = TextStyle(fontSize = 12.sp),
            modifier = Modifier.alignBy(LastBaseline)
          )
        }
      }
    }
  }

  example("NoriaSelectionContainer") {
    Row {
      NoriaSelectionContainer {
        Column {
          BasicText("Text 1")
          BasicText("Text 2")
          BasicText("טקסט 3")
        }
      }
    }
  }

  example("PartiallySelectableText") {
    Column {
      NoriaSelectionContainer {
        Column {
          BasicText("This text is selectable")
          BasicText("This one too")
          BasicText("This one as well")
          DisableSelection {
            BasicText("But not this one")
            BasicText("Neither this oneBasic")
          }
          BasicText("But again, you can select this one")
          BasicText("And this one too")
        }
      }
    }
  }

  val text = buildAnnotatedString {
    append("Simple text ")
    withAnnotation(tag = "[CLICK]", annotation = "[CLICKED1!]") {
      append("click me!")
    }
    append(" ")
    withAnnotation(tag = "[HOVER]", annotation = "[HOVERED1]") {
      append("hover me!")
    }
    append(" Some more text")
  }

  example("Clickable Text With Selection Without Column") {
    NoriaSelectionContainer {
      val onClick by rememberUpdatedState<(Int) -> Unit> { offset ->
        val annotations = text.getStringAnnotations("[CLICK]", offset, offset)
        println("Clicked at ${offset} ${annotations}")
      }
      val onHover by rememberUpdatedState<(Int?) -> Unit> { offset ->
        val annotations = offset?.let {
          text.getStringAnnotations("[HOVER]", offset, offset)
        }
        println("Hovered at ${offset} ${annotations}")
      }
      ClickableText(text = text + AnnotatedString(" [Selectable]"),
                    onClick = onClick,
                    onHover = onHover)
    }
  }

  example("Text With Compose Link") {
    NoriaSelectionContainer {
      val fontRasterizationSettings = LocalFontRasterizationSettings.current
      val theme = LocalTheme.current
      UiText(buildAnnotatedString {
        append("Text with ")
        link("Click me!",
             linkAnnotation = LinkAnnotation.Url("https://jetbrains.com"),
             theme = theme,
             fontRasterizationSettings = fontRasterizationSettings)
        append(" link wrapped in green theming.")
      }, textStyleKey = TextStyleKeys.Default)
    }
  }

  @Composable
  fun NoriaContext.LinkWithDynamciText(counter: Int) {
    val fontRasterizationSettings = LocalFontRasterizationSettings.current
    val theme = LocalTheme.current
    NoriaSelectionContainer {
      UiText(buildAnnotatedString {
        append("Text with ")
        link("Click me ${counter}!",
          linkAnnotation = LinkAnnotation.Url("https://jetbrains.com"),
          theme = theme,
          fontRasterizationSettings = fontRasterizationSettings)
        append(" link wrapped in green theming.")
      }, textStyleKey = TextStyleKeys.Default)

    }
  }

  example("Link With Dynamic Text") {
    val counterFlow = remember {
      flow {
        var i = 0
        while (true) {
          emit(i)
          delay(1000)
          i += 1
        }
      }
    }
    val counter by counterFlow.noriaCollectAsState(this, 0)
    LinkWithDynamciText(counter)
  }

  example("Very Long Link with Wraps") {
    /*
     * Knwon limtation: in current Compose implementation links can't properly handle wrapping.
     * Bassicaly, the bbox of link covers more space than needed when it's wrapped.
     */
    val text = "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida massa laoreet ultrices porttitor."
    val fontRasterizationSettings = LocalFontRasterizationSettings.current
    val theme = LocalTheme.current
    NoriaSelectionContainer {
      UiText(buildAnnotatedString {
        append("Text with $text")
        link("Click me!",
             linkAnnotation = LinkAnnotation.Url("https://jetbrains.com"),
             theme = theme,
             fontRasterizationSettings = fontRasterizationSettings)
        append("$text link wrapped in green theming.")
      }, textStyleKey = TextStyleKeys.DefaultMultiline, paragraphStyle = NoriaParagraphStyle.multiline)
    }
  }

  example("Lines Shouldn't Overlap") {
    Box {
      UiText(buildThemedAnnotatedString {
        text("Introducing\nJetBrains AI Assistant\n", textStyleKey = TextStyleKeys.Header2SemiBold)
      }, textStyleKey = TextStyleKeys.Header2SemiBold, paragraphStyle = NoriaParagraphStyle.multiline)
    }
  }
}