package noria.ui.examples

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.FontHinting
import androidx.compose.ui.text.FontRasterizationSettings
import androidx.compose.ui.text.FontSmoothing
import androidx.compose.ui.text.style.BaselineShift
import androidx.compose.ui.unit.*
import fleet.compose.theme.*
import fleet.compose.theme.components.Trim
import fleet.compose.theme.components.UiText
import fleet.compose.theme.components.buildThemedAnnotatedString
import fleet.compose.theme.components.buildStringWithMatcher
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.graphics.applyAlpha
import noria.NoriaContext
import noria.memo
import noria.state
import noria.ui.components.*
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery
import fleet.compose.theme.graphics.css
import noria.ui.components.split.SplitDirection
import noria.ui.components.split.splitView
import noria.ui.core.*
import noria.ui.text.*
import fleet.compose.theme.text.*
import fleet.compose.theme.keys.TextStyleKeys
import fleet.compose.theme.keys.ThemeKeys
import fleet.compose.theme.text.FontSpec
import fleet.compose.theme.text.NoriaTextDecoration
import fleet.compose.theme.util.ceil
import noria.ui.withModifier
import org.jetbrains.skia.Paint
import org.jetbrains.skia.TextBlob
import org.jetbrains.skia.impl.use
import org.jetbrains.skia.shaper.Shaper
import org.jetbrains.skia.shaper.ShapingOptions
import org.jetbrains.skia.shaper.TextBlobBuilderRunHandler

internal fun textExamples(): Gallery = gallery("Text", NoriaExamples.sourceCodeForFile("Text.kt")) {
  example("Regular UI Text") {
    UiText("Text")
  }

  example("Trimmed Text") {
    Box(contentAlignment = Alignment.TopStart) {
      withModifier(Modifier.width(width = 120.dp)) {
        Column {
          UiText("Some text with an ellipsis in the middle", trim = Trim.Middle)
          UiText("Some text with an ellipsis in the end", trim = Trim.End)
          UiText("Some text with an ellipsis in the end and underscored",
                 noriaTextDecoration = NoriaTextDecoration(style = NoriaTextDecoration.Style.SOLID),
                 trim = Trim.End)
        }
      }
    }
  }

  example("Text Rendering Playground") {
    val labelWidth = 150.dp

    @Composable
    fun <T> ColumnScope.ParameterInput(label: String, initialValue: T): String {
      val textInputModel = textInputModel(initialValue?.toString() ?: "")
      Row {
        UiText(label, modifier = Modifier.width(labelWidth))
        textInput(textInputModel)
      }
      return textInputModel.state.read().content
    }

    @Composable
    fun <T> ColumnScope.ParameterDropDown(label: String, items: List<T>, initialValue: T, presentationFn: (T) -> String): T {
      val state = state { initialValue }
      Row {
        UiText(label, modifier = Modifier.width(labelWidth))
        dropDownList(items, state.read(), presentationFn) { selection -> state.update { selection } }
      }
      return state.read()
    }

    Row {
      Column(
        Modifier.padding(4.dp, 8.dp),
        verticalArrangement = Arrangement.spacedBy(4.dp),
      ) {
        val sample = ParameterInput("Sample", "The quick brown fox jumps over the lazy dog")
        val textSpec = textSpec(TextStyleKeys.Default)
        val fontFamily = ParameterInput("Font Family", textSpec.fontSpec.family)
        val fontSize = ParameterInput("Font Size", textSpec.fontSpec.size.value).toFloatOrNull()?.sp ?: 13.sp
        val weight = ParameterInput("Weight", textSpec.fontSpec.weight).toIntOrNull()?.coerceIn(1, 1000) ?: 400
        val italicState = state { false }
        Row {
          UiText("Italic", modifier = Modifier.width(labelWidth))
          Checkbox(italicState.read(), { checked -> italicState.update { checked } })
        }
        val lineHeight = ParameterInput("Line Height", textSpec.lineHeight).toFloatOrNull()
        val baselineShift = ParameterInput("Baseline Shift", textSpec.baselineShift.multiplier).toFloatOrNull() ?: 0f
        val platformFontRasterizationSettings = LocalFontRasterizationSettings.current
        val hinting = ParameterDropDown("Hinting", FontHinting.entries.toList(), platformFontRasterizationSettings.hinting) { it.name }
        val smoothing = ParameterDropDown("Smoothing", FontSmoothing.entries.toList(), platformFontRasterizationSettings.smoothing) { it.name }
        val autoHintingForcedState = state { false }
        Row {
          UiText("Auto-Hinting Forced", modifier = Modifier.width(labelWidth))
          Checkbox(autoHintingForcedState.read(), { checked -> autoHintingForcedState.update { checked } })
        }
        val fontRasterizationSettings = FontRasterizationSettings(
          smoothing = smoothing,
          hinting = hinting,
          subpixelPositioning = platformFontRasterizationSettings.subpixelPositioning,
          autoHintingForced = autoHintingForcedState.read(),
        )
        CompositionLocalProvider(LocalFontRasterizationSettings provides fontRasterizationSettings) {
          UiText(
            sample,
            userTextSpec = textSpec.copy(
              fontSpec = FontSpec(
                family = fontFamily,
                size = fontSize,
                weight = weight,
                italic = italicState.read(),
              ),
              lineHeight = lineHeight,
              baselineShift = BaselineShift(baselineShift),
            ),
          )
        }
      }
    }
  }

  example("Fleet UI Kit Typography") {
    val fonts = listOf(
      "Header 1" to TextStyleKeys.Header1,
      "Header 2" to TextStyleKeys.Header2,
      "Header 2 bold" to TextStyleKeys.Header2Bold,
      "Header 3" to TextStyleKeys.Header3,
      "Header 3 bold" to TextStyleKeys.Header3Bold,
      "Default" to TextStyleKeys.Default,
      "Default bold" to TextStyleKeys.DefaultBold,
      "Default multiline" to TextStyleKeys.DefaultMultiline,
      "Medium" to TextStyleKeys.Medium,
      "Medium bold" to TextStyleKeys.MediumBold,
      "Small" to TextStyleKeys.Small,
      "Code" to TextStyleKeys.Code,
      "Code small" to TextStyleKeys.CodeSmall,
    )

    Row {
      Column {
        fonts.forEach { (text, key) ->
          Row(Modifier.padding(4.dp, 8.dp), verticalAlignment = Alignment.CenterVertically) {
            val density = LocalDensity.current
            val textSpec = textSpec(key)
            val baselineShiftModel = textInputModel(textSpec.baselineShift.multiplier.toString())
            val baselineShift = baselineShiftModel.state.read().content.toFloatOrNull()?.let {
              BaselineShift(it)
            } ?: BaselineShift.None
            val textSpecWithBaseline = textSpec.copy(baselineShift = baselineShift)
            val annotatedText = buildThemedAnnotatedString {
              text(text, userTextSpec = textSpecWithBaseline, backgroundColor = Color.LightGray.applyAlpha(0.3f))
            }

            UiText(annotatedText)
            Spacer(Modifier.width(8.dp))
            //            CullingLayout {
            //              layout(
            //                width = with(density) { 200.dp.roundToPx() },
            //                height = with(density) { 16.dp.roundToPx() }) {
            //                val textHeight = with(density) { paragraph.height.toDp() }
            //                UiText("${textSpec.fontSpec.size} × ${textSpec.lineHeight} = ${textHeight}")
            //              }
            //            }
            //            Spacer(Modifier.width(8.dp))
            UiText("Baseline Shift:")
            Spacer(Modifier.width(4.dp))
            textInput(baselineShiftModel, style = innerTextInputStyle())
          }
        }
      }
    }
  }


  example("Emoji") {
    UiText("""😁🔥🥦👮🏼""")
  }

  example("Custom Color") {
    UiText("Colored text", color = theme[ThemeKeys.DangerousButtonTextDefault])
  }

  example("Custom Spec") {
    UiText("Bold text", textStyleKey = TextStyleKeys.DefaultBold)
  }

  val loremIpsum = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum."

  example("Wrapping Text in a Box") {
    UiText(text = loremIpsum,
           modifier = Modifier.widthIn(max = 400.dp),
           paragraphStyle = NoriaParagraphStyle.multiline)
  }

  example("Multiline Text") {
    splitView(direction = SplitDirection.HORIZONTAL) {
      first {
        UiText(loremIpsum, paragraphStyle = NoriaParagraphStyle.multiline)
      }
      second {
        UiText(loremIpsum, paragraphStyle = NoriaParagraphStyle.multiline)
      }
    }
  }

  example("Different Line Height") {
    val text = "First line\nSecond line"
    val textSpec = textSpec(TextStyleKeys.Default)

    @Composable
    fun RowScope.withHeight(height: Float?) {
      Column(modifier = Modifier.padding(all = 4.dp)) {
        UiText("height: $height")
        Spacer(Modifier.width(4.dp))
        UiText(text,
               userTextSpec = textSpec.copy(lineHeight = height),
               paragraphStyle = NoriaParagraphStyle.multiline)
      }
    }
    Row {
      withHeight(null)
      withHeight(0f)
      withHeight(1f)
      withHeight(1.2f)
      withHeight(1.3f)
    }
  }

  @Composable
  fun NoriaContext.textLine(
    text: String,
    fontSpec: FontSpec,
    color: Color = theme[ThemeKeys.TextPrimary],
    textDecoration: NoriaTextDecoration? = null,
  ) {
    layout { constraints ->
      val font = findFont(fontSpec)
      val fgColor = color.applyAlpha(contrast)
      val shaper = Shaper.makeShapeThenWrap()
      val handler = TextBlobBuilderRunHandler<Any>(text)
      shaper.shape(text, font, ShapingOptions.DEFAULT, constraints.maxWidth.toFloat(), handler)
      val blob: TextBlob? = handler.makeBlob()
      blob?.let {
        val lineWidth = blob.blockBounds.width.ceil()
        val left = 0
        val boxHeight = blob.blockBounds.height.ceil()
        render(IntRect(IntOffset.Zero, constraints.constrain(IntSize(lineWidth, boxHeight)))) {
          Paint().use { paint ->
            paint.color4f = fgColor.toSkija()
            withCanvas { canvas -> canvas.nativeCanvas.drawTextBlob(blob, 0f, 0f, paint) }
          }

          if (textDecoration != null) {
            renderDecoration(decoration = textDecoration,
                             metrics = font.textDecorationMetrics(),
                             left = left,
                             right = left + lineWidth,
                             fallbackColor = fgColor)
          }
        }
      } ?: IntSize.Zero
    }
  }

  example("Font Fallbacks") {
    val text = "Hello world こんにちは世界"
    Column {
      UiText(text, textStyleKey = TextStyleKeys.Code)
      textLine(text, fontSpec = textSpec(TextStyleKeys.Default).fontSpec, textDecoration = NoriaTextDecoration(Color.Red))
      textLine("שר-++ה", fontSpec = textSpec(TextStyleKeys.Code).fontSpec)
      textLine("שר-++ה", fontSpec = textSpec(TextStyleKeys.Default).fontSpec)
      textLine("Sarah (שר--+ה)", fontSpec = textSpec(TextStyleKeys.Code).fontSpec)
      textLine("Sarah (שר--+ה)", fontSpec = textSpec(TextStyleKeys.Default).fontSpec)
    }
  }

  example("Diacritics") {
    Column {
      UiText("Ex\u00E1mple (u00E1)", textStyleKey = TextStyleKeys.Header3Bold)
      UiText("Ex\u0061\u0301mple (u0061u0301)", textStyleKey = TextStyleKeys.Header3Bold)
    }
  }

  example("Different Font Rendering Settings") {
    // Now only low level text rendering api respects font rendering settings
    val textSpecState = state { textSpec(TextStyleKeys.Code) }
    val fontSpec = textSpecState.read().fontSpec
    // defaults for my mac is:
    // FontHinting.NORMAL
    val hintingOptions = listOf(FontHinting.None, FontHinting.Slight, FontHinting.Normal, FontHinting.Full)
    // false
    val isAutoHintingForcedOptions = listOf(false, true)
    // false
    val subpixelOptions = listOf(false, true)
    // FontEdging.ANTI_ALIAS
    val edgingOptions = listOf(FontSmoothing.None, FontSmoothing.AntiAlias, FontSmoothing.SubpixelAntiAlias)
    val text = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua."
    Column {
      dropDownList(memo { listOf(textSpec(TextStyleKeys.Default), textSpec(TextStyleKeys.Code)) },
                   textSpecState,
                   presentationFn = { if (it == textSpec(TextStyleKeys.Default)) "Default" else "Code" })
      for (subpixel in subpixelOptions) {
        for (edging in edgingOptions) {
          for (hinting in hintingOptions) {
            for (autoHintingForced in isAutoHintingForcedOptions) {
              val rasterSettings = FontRasterizationSettings(subpixelPositioning = subpixel,
                                                             smoothing = edging,
                                                             hinting = hinting,
                                                             autoHintingForced = autoHintingForced)
              CompositionLocalProvider(LocalFontRasterizationSettings provides rasterSettings) {
                UiText("$rasterSettings")
              }
              CompositionLocalProvider(LocalFontRasterizationSettings provides rasterSettings) {
                textLine(text, fontSpec = fontSpec)
              }
              Spacer(Modifier.height(8.dp))
            }
          }
        }
      }
    }
  }

  @Composable
  fun NoriaContext.fakeUnderline(text: String, textColor: Color, modifier: Modifier) {
    UiText(buildStringWithMatcher(text = text,
                                  matcher = null,
                                  textColor = textColor,
                                  decoration = NoriaTextDecoration(Color.Transparent, style = NoriaTextDecoration.Style.SOLID, thicknessMultiplier = 1f)),
           modifier = modifier,
           paragraphStyle = NoriaParagraphStyle.oneLine)
  }

  example("Transparent Underline") {
    // we expect that underline will be transparent, but it's white apparently
    fakeUnderline("Hello!", textColor = Color.Black, modifier = Modifier)
  }

  example("Text Code") {
    Column {
      UiText("Some Text!!!", textStyleKey = TextStyleKeys.Code, color = theme[ThemeKeys.TextSecondary])
      UiText("Some Unexpected Font", userTextSpec = TextSpec(FontSpec("Unexpected Font", 14.sp)))
    }
  }

  example("SmallItalic") {
    UiText("Some strange text", color = Color.Gray, textStyleKey = TextStyleKeys.SmallItalic)
  }

  example("Short Trimmed Text") {
    UiText("1", trim = Trim.Middle)
  }

  example("Line Height for Single Line") {
//    val text = "First line"
//    val textSpec = textSpec(TextStyleKeys.Default)
//
//    @Composable
//    fun RowScope.withHeight(height: Float?) {
//      Column(modifier = Modifier.padding(all = 4.dp)) {
////        UiText("height: $height")
////        Spacer(Modifier.width(4.dp))
//        UiText(text,
//               backgroundColor = Color.Gray,
//               userTextSpec = textSpec.copy(lineHeight = height),
//               paragraphStyle = NoriaParagraphStyle.oneLine)
//      }
//    }
//    Row {
//      withHeight(null)
//      withHeight(0f)
//      withHeight(1f)
//      withHeight(1.2f)
//      withHeight(1.3f)
//    }

    Column(modifier = Modifier.background(Color.White),
           verticalArrangement = Arrangement.spacedBy(10.dp)) {
      UiText(text = "Small",
             color = Color.White,
             backgroundColor = Color.css("2D2D2D"),
             userTextSpec = TextSpec(fontSpec = FontSpec("Inter", size = 10.sp),
                                     lineHeight = 14f / 10f))
      UiText(text = "Default",
                   color = Color.White,
                   backgroundColor = Color.css("2D2D2D"),
                   userTextSpec = TextSpec(fontSpec = FontSpec("Inter", size = 13.sp),
                                           lineHeight = 16f / 13f))
    }
  }
}
