package noria.ui.examples

import androidx.compose.foundation.focusGroup
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.unit.dp
import fleet.compose.foundation.focusTraversalScope
import fleet.compose.foundation.nonFocusable
import fleet.compose.theme.components.*
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.components.gallery.gallery
import noria.ui.components.textInput
import noria.ui.text.Link

@OptIn(ExperimentalComposeUiApi::class)
internal fun focusTraversalExamples() = gallery("Focus Traversal", NoriaExamples.sourceCodeForFile("FocusTraversal.kt")) {
  example("Simple") {
    val focusRequester = remember { FocusRequester() }
    Column {
      Button(text = "Focus") { focusRequester.requestFocus() }
      Spacer(Modifier.height(16.dp))
      Column(Modifier.focusRequester(focusRequester).focusTraversalScope()) {
        textInput("1")
        Spacer(Modifier.height(8.dp))
        textInput("2")
        Spacer(Modifier.height(8.dp))
        textInput("3")
      }
    }
  }

  /**
   * UI Components bring their own built-in focus targets
   */
  example("Components") {
    val focusRequester = remember { FocusRequester() }
    Column {
      Button(text = "Focus") { focusRequester.requestFocus() }
      Spacer(Modifier.height(16.dp))
      Column(Modifier.focusRequester(focusRequester).focusTraversalScope()) {
        textInput("1")
        Spacer(Modifier.height(8.dp))
        Button(text = "2") { }
        Spacer(Modifier.height(8.dp))
        Link("3") { }
        Spacer(Modifier.height(8.dp))
        DangerousButton(text = "4") { }
        Spacer(Modifier.height(8.dp))
        Column {
          SmallPrimaryButton(text = "5") { }
          Spacer(Modifier.height(8.dp))
          SmallButton(text = "6") { }
        }
        Spacer(Modifier.height(8.dp))
        val toggleButtonState = remember { mutableStateOf(false) }
        ToggleButton("7", selected = toggleButtonState.value, onClick = {
          toggleButtonState.value = !toggleButtonState.value
        })
        Spacer(Modifier.height(8.dp))
        var checkboxState by remember { mutableStateOf(false) }
        Checkbox(checkboxState, label = "8", onCheckedChange = { checkboxState = it })
      }
    }
  }

  example("Tree With Inner Root") {
    val focusRequester = remember { FocusRequester() }
    Column {
      Button(text = "Focus") { focusRequester.requestFocus() }
      Spacer(Modifier.height(16.dp))
      Column(Modifier.focusRequester(focusRequester).focusTraversalScope().fillMaxWidth(), propagateMinWidth = true) {
        textInput("1")
        Spacer(Modifier.height(8.dp))
        Row(Modifier.focusRequester(focusRequester).focusTraversalScope()) {
          textInput("2.1", Modifier.weight(1f))
          Spacer(Modifier.width(8.dp))
          textInput("2.2", Modifier.weight(1f))
          Spacer(Modifier.width(8.dp))
          textInput("2.3", Modifier.weight(1f))
        }
        Spacer(Modifier.height(8.dp))
        textInput("3")
      }
    }
  }

  /**
   * Focus traversal skips all disabled nodes and their descendants, even if the descendants are enabled
   */
  example("Disabled Subtree") {
    Column(Modifier.focusTraversalScope().fillMaxWidth(), propagateMinWidth = true) {
      textInput("1")
      Spacer(Modifier.height(8.dp))
      Row(Modifier.nonFocusable()) {
        Button(Modifier.weight(1f), text = "2.1") {}
        Spacer(Modifier.width(8.dp))
        Button(Modifier.weight(1f), text = "2.2") {}
        Spacer(Modifier.width(8.dp))
        Button(Modifier.weight(1f), text = "2.3") {}
      }
      Spacer(Modifier.height(8.dp))
      textInput("3")
    }
  }

  /**
   * Focus traversal skips the whole subtree when all its leaves are disabled
   */
  example("Exclusively Disabled Leaves in Subtree") {
    Column(Modifier.focusTraversalScope().fillMaxWidth(), propagateMinWidth = true) {
      textInput("1")
      Spacer(Modifier.height(8.dp))
      Row(Modifier.focusGroup()) {
        Button(Modifier.weight(1f), text = "2.1", enabled = false) {}
        Spacer(Modifier.width(8.dp))
        Button(Modifier.weight(1f), text = "2.2", enabled = false) {}
        Spacer(Modifier.width(8.dp))
        Button(Modifier.weight(1f), text = "2.3", enabled = false) {}
      }
      Spacer(Modifier.height(8.dp))
      textInput("3")
    }
  }
}