package noria.ui.examples

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import fleet.compose.theme.components.checkbox.Checkbox
import fleet.compose.theme.components.checkbox.CheckboxState
import fleet.compose.theme.components.checkbox.TriStateCheckbox
import noria.scope
import noria.state
import fleet.compose.theme.components.gallery.Gallery
import fleet.compose.theme.components.gallery.gallery

internal fun checkboxExamples(): Gallery = gallery("Checkbox", NoriaExamples.sourceCodeForFile("Checkbox.kt")) {
  example("Simple Checkbox") {
    Column {
      var checked by remember { mutableStateOf(CheckboxState.Checked) }
      var partial by remember { mutableStateOf(CheckboxState.Partial) }
      var unchecked by remember { mutableStateOf(CheckboxState.Unchecked) }
      val modifier = Modifier.padding(vertical = 4.dp)
      TriStateCheckbox(checked, label = "checked", modifier = modifier, onClick = { checked = checked.toggle() })
      TriStateCheckbox(partial, label = "partial", modifier = modifier, onClick = { partial = partial.toggle() })
      TriStateCheckbox(unchecked, label = "unchecked", modifier = modifier, onClick = { unchecked = unchecked.toggle() })
    }
  }
  example("Disabled Checkbox") {
    Column {
      val modifier = Modifier.padding(vertical = 4.dp)
      TriStateCheckbox(CheckboxState.Checked, label = "checked", enabled = false, modifier = modifier,
                       onClick = { error("Should not call onChange") })
      TriStateCheckbox(CheckboxState.Partial, label = "partial", enabled = false, modifier = modifier,
                       onClick = { error("Should not call onChange") })
      TriStateCheckbox(CheckboxState.Unchecked, label = "partial", enabled = false, modifier = modifier,
                       onClick = { error("Should not call onChange") })
    }
  }
  example("Checkbox Tree", preferredHeight = 400.dp) {
    val childLabels = listOf("foo", "baz", "bar", "qux")
    val selectionState = state { childLabels.associateWith { false } }
    val checkboxPaddingModifier = Modifier.padding(vertical = 4.dp)
    Column {
      Row {
        TriStateCheckbox(
          state = when (selectionState.read().values.count { it }) {
            childLabels.size -> CheckboxState.Checked
            0 -> CheckboxState.Unchecked
            else -> CheckboxState.Partial
          },
          onClick = {
            selectionState.update { selectionMap ->
              if (selectionMap.values.all { it }) {
                childLabels.associateWith { false }
              }
              else {
                childLabels.associateWith { true }
              }
            }
          },
          label = "root",
          modifier = checkboxPaddingModifier,
        )
      }

      Column(modifier = Modifier.padding(horizontal = 4.dp)) {
        childLabels.forEachIndexed { i, label ->
          scope(i) {
            Row {
              Checkbox(checked = selectionState.read()[label] ?: false,
                       onCheckedChange = {
                         selectionState.update { selectionMap ->
                           selectionMap + (label to (selectionMap[label] != true))
                         }
                       },
                       label = label,
                       modifier = checkboxPaddingModifier)
            }
          }
        }
      }
    }
  }
}