import type * as ts from "./tsserverlibrary.shim"
import type {
  GetElementTypeArguments,
  GetElementTypeResponse,
  GetSymbolTypeArguments,
  GetSymbolTypeResponse,
  GetTypePropertiesArguments
} from "./protocol"
import {getElementTypeByOffsets, getSymbolType, getTypeProperties} from "./ide-get-element-type"

const decoratedLanguageServices = new WeakSet();

export function decorateLanguageService(languageService: ts.LanguageService) {
  if (decoratedLanguageServices.has(languageService)) {
    return
  }
  decoratedLanguageServices.add(languageService);

  languageService.webStormGetElementType = (ts, fileName, startOffset, endOffset, forceReturnType, cancellationToken, reverseMapper) => {
    // see getQuickInfoAtPosition
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined
    }

    return getElementTypeByOffsets(ts, languageService.ideProjectId, program, sourceFile, startOffset, endOffset,
                                   forceReturnType, cancellationToken, reverseMapper)
  }

  languageService.webStormGetSymbolType = (ts, symbolId, cancellationToken, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getSymbolType(ts, program, symbolId, cancellationToken, reverseMapper)
  }

  languageService.webStormGetTypeProperties = (ts, typeId, cancellationToken, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeProperties(ts, program, typeId, cancellationToken, reverseMapper)
  }
}

export function getElementTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                       projectService: ts.server.ProjectService,
                                       request: ts.server.protocol.Request): GetElementTypeResponse {
  const requestArguments = request.arguments as GetElementTypeArguments

  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let {project, sourceFile} = projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file,
                                                                                       requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined

  let range = requestArguments.range
  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  return project.getLanguageService().webStormGetElementType(ts, fileName, startOffset, endOffset, requestArguments.forceReturnType,
                                                             projectService.cancellationToken)
}

export function getSymbolTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                      projectService: ts.server.ProjectService,
                                      request: ts.server.protocol.Request): GetSymbolTypeResponse {
  const requestArguments = request.arguments as GetSymbolTypeArguments

  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetSymbolType(ts, requestArguments.symbolId, projectService.cancellationToken)
}

export function getTypePropertiesTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                          projectService: ts.server.ProjectService,
                                          request: ts.server.protocol.Request): GetElementTypeResponse {
  const requestArguments = request.arguments as GetTypePropertiesArguments

  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetTypeProperties(ts, requestArguments.typeId, projectService.cancellationToken)
}


function findLanguageService(projectService: ts.server.ProjectService, ideProjectId: number, ideTypeCheckerId: number): ts.LanguageService | undefined {
  for (let [, project] of projectService.configuredProjects as Map<string, ts.server.ConfiguredProject>) {
    if (project.ideProjectId === ideProjectId) {
      return getLanguageService(project)
    }
  }

  for (let inferredProject of projectService.inferredProjects) {
    if (inferredProject.ideProjectId === ideProjectId) {
      return getLanguageService(inferredProject)
    }
  }

  for (let externalProject of projectService.externalProjects) {
    if (externalProject.ideProjectId === ideProjectId) {
      return getLanguageService(externalProject)
    }
  }

  function getLanguageService(project: ts.server.Project) {
    let program = project.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return project.getLanguageService()
    }
    console.error(`findLanguageService - failed to find language service for ideProjectId ${ideProjectId} - wrong typeCheckerId. Cached: ${program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId}, requested: ${ideTypeCheckerId}`)
  }

  console.error(`findLanguageService - failed to find language service for ideProjectId ${ideProjectId}`)
}

