import {
  GetElementTypeArguments,
  GetElementTypeResponse,
  GetSymbolTypeArguments,
  GetSymbolTypeResponse,
  GetTypePropertiesArguments,
  Range
// @ts-ignore
} from "tsc-ide-plugin/protocol"
// @ts-ignore
import {getElementType, getSymbolType, getTypeProperties, ReverseMapper} from "tsc-ide-plugin/ide-get-element-type"
import {type DocumentsAndMap, getGeneratedRange, getSourceRange} from '@volar/language-service/lib/utils/featureWorkers';
import {getDocs, getLanguageService, asFileName, asUri} from "./volar-utils";
import type {LanguageServiceContext} from '@volar/language-service';
import {URI} from 'vscode-uri';
import type ts from "typescript";

export async function getElementTypeVolarServer(ts: typeof import("typescript"),
                                                context: LanguageServiceContext,
                                                uri: string,
                                                requestArguments: GetElementTypeArguments): Promise<GetElementTypeResponse> {
  const range: Range = requestArguments.range;
  return await callInMapper(context, uri, range, (program, sourceFile, range, reverseMapper) => {
    return getElementType(ts, program, sourceFile, range, requestArguments.forceReturnType, reverseMapper);
  })
}

export async function getSymbolTypeVolarServer(ts: typeof import("typescript"),
                                               context: LanguageServiceContext,
                                               uri: string,
                                               requestArguments: GetSymbolTypeArguments): Promise<GetSymbolTypeResponse> {
  return await callInMapper(context, uri, undefined, (program, sourceFile, range, reverseMapper) => {
    return getSymbolType(ts, program, requestArguments.symbolId, reverseMapper);
  })
}

export async function getTypePropertiesVolarServer(ts: typeof import("typescript"),
                                                   context: LanguageServiceContext,
                                                   uri: string,
                                                   requestArguments: GetTypePropertiesArguments): Promise<GetElementTypeResponse> {
  return await callInMapper(context, uri, undefined, (program, sourceFile, range, reverseMapper) => {
    return getTypeProperties(ts, program, requestArguments.typeId, reverseMapper);
  })
}


async function callInMapper<T>(
  context: LanguageServiceContext,
  _uri: string,
  _range: Range | undefined,
  callable: (program: ts.Program, sourceFile: ts.SourceFile, range: Range | undefined, reverseMapper: ReverseMapper) => T,
): Promise<T | undefined> {
  if (context.project.typescript == null) return undefined;

  const uri = URI.parse(_uri);

  let program = getLanguageService(context)?.getProgram();
  if (!program) return undefined;

  const fileName = asFileName(context.project.typescript, uri)
  const sourceFile = program.getSourceFile(fileName)
  if (!sourceFile) return undefined;

  let docs = getDocs(context, uri);
  let range: Range | undefined = _range;
  if (docs && _range) {
    range = getGeneratedRange(docs, _range);
  }

  const reverseMapper = (targetFile: ts.SourceFile, generatedRange: Range) => {
    const targetName = targetFile.fileName;
    if (sourceFile == targetFile) {
      return doReverseMap(docs, generatedRange, targetName);
    }
    else {
      const targetUri = asUri(context.project.typescript!, targetName);
      const docs = getDocs(context, targetUri);
      return doReverseMap(docs, generatedRange, targetName);
    }
  }

  return callable(program, sourceFile, range, reverseMapper);
}

function doReverseMap(docs: DocumentsAndMap | undefined, generatedRange: Range, targetName: string) {
  if (docs == null) return undefined;

  const sourceRange = getSourceRange(docs, generatedRange);
  if (sourceRange != undefined) {
    const sourceDocument = docs[0];
    return {
      pos: sourceDocument.offsetAt(sourceRange.start),
      end: sourceDocument.offsetAt(sourceRange.end),
      fileName: targetName
    }
  }
}