// Copyright 2016-2022 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.goide.project;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.ThreeState;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public interface GoRootsProvider {
  ExtensionPointName<GoRootsProvider> EP_NAME = ExtensionPointName.create("com.goide.rootsProvider");

  /**
   * Returns all GOPATH paths or default one ({@link #getDefaultGoPath(Project, Module)}).
   * These paths will be passed to terminal and to all external go processes as a GOPATH env variable.
   * <p>
   * Default implementation of the extension returns paths based on user settings at Go | GOPATH.
   */
  @NotNull
  Collection<VirtualFile> getGoPathRoots(@Nullable Project project, @Nullable Module module);

  /**
   * Returns all paths to Go sources directories.
   * These paths will be used for indexing, attached as External libraries.
   * Also, import paths of a particular file or directory will be calculated relatively to these paths.
   * <p>
   * Default implementation of the extension returns all `src`-children of {@link #getGoPathRoots(Project, Module)} paths
   */
  @NotNull
  Collection<VirtualFile> getGoPathSourcesRoots(@Nullable Project project, @Nullable Module module);

  /**
   * Returns all paths to Go-related bin directories.
   * These paths will be passed to terminal and to all external go processes as a part of PATH env variable,
   * and also will be used to looking for go-related binary like `goimport`, `gotype`, etc.
   * <p>
   * Default implementation of the extension returns $GOBIN path, all `bin`-children of {@link #getGoPathRoots(Project, Module)} paths and bin directory of GOROOT
   */
  @NotNull
  Collection<VirtualFile> getGoPathBinRoots(@Nullable Project project, @Nullable Module module);

  /**
   * Returns true if roots might be located outside project and must be attached as a libraries.
   * <p>
   * External providers are used for building indices, so the method must return true
   * if a provider accesses to any indices.
   * <p>
   * E.g. AppEngine roots provider can return only project-local directories.
   */
  boolean isExternal();

  /**
   * @return whether IDE should build index for entire GOPATH.
   * <p>
   * First non-unsure response will be considered as a result.
   */
  default @NotNull ThreeState indexGoPathSources(@NotNull Project project) {
    return ThreeState.UNSURE;
  }

  /**
   * Returns the default GOPATH value.
   * <p>
   * This path is usually used if GOPATH configuration is absent. E.g. since go1.8, the default GOPATH is `~/go`
   */
  @Nullable
  default VirtualFile getDefaultGoPath(@Nullable Project project, @Nullable Module module) {
    return null;
  }

  /**
   * @return vendor directories that can contain declarations resolvable from a {@code file},
   * or @{code null} if provider cannot calculate such directories.
   */
  @Nullable
  default Collection<VirtualFile> getVendorDirectoriesInResolveScope(@NotNull Project project,
                                                                     @Nullable Module module,
                                                                     @Nullable VirtualFile file) {
    return null;
  }
}
