// noinspection JSBitwiseOperatorUsage

import type * as ts from "typescript/lib/tsserverlibrary"
import type {GetElementTypeResponse, GetSymbolTypeResponse, Range} from "./protocol"

declare global {
  type RegularMap<K, V> = Map<K, V>
  namespace ts {
    interface TypeChecker {
      webStormCacheInfo?: {
        ideTypeCheckerId: number,
        requestedTypeIds: Set<number>,
        seenTypeIds: RegularMap<number, Type>,
        seenSymbolIds: RegularMap<number, Symbol>,
      }
    }

    interface Type {
      id: number | undefined // internal
    }
  }
}

type TS = typeof import('typescript/lib/tsserverlibrary')
export type ReverseMapper = (sourceFile: ts.SourceFile, generatedRange: Range) => any;

let lastIdeTypeCheckerId: number = 0

export function getElementType(
  ts: TS,
  program: ts.Program,
  sourceFile: ts.SourceFile,
  range: Range,
  forceReturnType: boolean,
  reverseMapper?: ReverseMapper,
): GetElementTypeResponse {
  let positionStart = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let positionEnd = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  let node: ts.Node = (ts as any).getTokenAtPosition(sourceFile, positionStart);
  while (node && node.getEnd() < positionEnd) {
    node = node.parent;
  }
  if (!node || node === sourceFile) {
    return undefined;
  }
  if ((ts.isStringLiteral(node) || ts.isNumericLiteral(node))
      && node.pos === node.parent?.pos
      && node.end === node.parent?.end) {
    node = node.parent;
  }

  const typeChecker = program.getTypeChecker();
  if (!typeChecker.webStormCacheInfo) {
    typeChecker.webStormCacheInfo = {
      ideTypeCheckerId: ++lastIdeTypeCheckerId,
      requestedTypeIds: new Set<number>(),
      seenTypeIds: new Map<number, ts.Type>(),
      seenSymbolIds: new Map<number, ts.Symbol>()
    }
  }
  const cacheInfo = typeChecker.webStormCacheInfo;

  let type = typeChecker.getTypeAtLocation(node);

  let prepared: Record<string, unknown>;
  if (forceReturnType || type.id == null || !cacheInfo.requestedTypeIds.has(type.id)) {
    const ctx: ConvertContext = {
      ts: ts,
      checker: typeChecker,
      createdObjectsIdeIds: new Map(),
      reverseMapper,
      level: 0,
      nextId: 0,
    };
    prepared = convertType(type, ctx) as Record<string, unknown>;
    prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
    if (type.id != null) {
      cacheInfo.requestedTypeIds.add(type.id);
    }
  }
  else {
    prepared = {
      id: type.id,
      ideTypeCheckerId: cacheInfo.ideTypeCheckerId,
    };
  }

  return {responseRequired: true, response: prepared}
}

type ConvertContext = {
  ts: TS,
  checker: ts.TypeChecker,
  createdObjectsIdeIds: Map<object, number>,
  reverseMapper?: ReverseMapper,
  level: number,
  nextId: number,
}

function convertType(
  type: ts.Type,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    type,
    ideObjectId => {
      const tscType: Record<string, unknown> = {
        ideObjectId,
        ideObjectType: "TypeObject",
        flags: type.flags,
      }

      // First, we map only types -- to get them on shallower levels, allowing them more room for nested items.
      // Then we map everything else -- these entries will likely get references to already mapped types.

      tscType.aliasTypeArguments = type.aliasTypeArguments
        ?.map(t => convertType(t, ctx))
        ?.filter(isNotNull);

      if (type.flags & ctx.ts.TypeFlags.Object) {
        if ((type as any).target)
          // FIXME internal API
          // FIXME Whole target is not needed, only target.objectFlags needed
          tscType.target = convertType((type as any).target, ctx);

        if ((type as ts.ObjectType).objectFlags & ctx.ts.ObjectFlags.Reference)
          tscType.resolvedTypeArguments = ctx.checker.getTypeArguments(type as ts.TypeReference)
            // TS 4 returns 'this'-type as one of class type arguments, which we don't need
            // FIXME internal API
            .filter(t => !(t as any).isThisType)
            .map((t: ts.Type) => convertType(t, ctx))
            .filter(isNotNull);
      }

      if (type.flags & (ctx.ts.TypeFlags.UnionOrIntersection | ctx.ts.TypeFlags.TemplateLiteral))
        tscType.types = (type as ts.UnionOrIntersectionType).types
          .map(t => convertType(t, ctx))
          .filter(isNotNull);

      if (type.flags & ctx.ts.TypeFlags.Literal && (type as ts.LiteralType).freshType)
        tscType.freshType = convertType((type as ts.LiteralType).freshType, ctx);

      if (type.flags & ctx.ts.TypeFlags.TypeParameter) {
        const constraint = ctx.checker.getBaseConstraintOfType(type);
        if (constraint) tscType.constraint = convertType(constraint, ctx);
      }

      if (type.flags & ctx.ts.TypeFlags.Index)
        tscType.type = convertType((type as ts.IndexType).type, ctx);

      if (type.flags & ctx.ts.TypeFlags.IndexedAccess) {
        tscType.objectType = convertType((type as ts.IndexedAccessType).objectType, ctx);
        tscType.indexType = convertType((type as ts.IndexedAccessType).indexType, ctx);
      }

      if (type.flags & ctx.ts.TypeFlags.Conditional) {
        tscType.checkType = convertType((type as ts.ConditionalType).checkType, ctx);
        tscType.extendsType = convertType((type as ts.ConditionalType).extendsType, ctx);

        ctx.checker.getPropertiesOfType(type); // In TS 4 this triggers calculation of true and false types
        if ((type as ts.ConditionalType).resolvedTrueType)
          tscType.resolvedTrueType = convertType((type as ts.ConditionalType).resolvedTrueType as ts.Type, ctx);
        if ((type as ts.ConditionalType).resolvedFalseType)
          tscType.resolvedFalseType = convertType((type as ts.ConditionalType).resolvedFalseType as ts.Type, ctx);
      }

      // Now map everything else but types

      if (type.symbol) tscType.symbol = convertSymbol(type.symbol, ctx);
      if (type.aliasSymbol) tscType.aliasSymbol = convertSymbol(type.aliasSymbol, ctx);

      if (type.flags & ctx.ts.TypeFlags.Object) {
        tscType.objectFlags = (type as ts.ObjectType).objectFlags;
      }

      if (type.flags & ctx.ts.TypeFlags.Literal) {
        if (type.flags & ctx.ts.TypeFlags.BigIntLiteral)
          tscType.value = convertPseudoBigInt((type as ts.BigIntLiteralType).value, ctx);
        else
          tscType.value = (type as ts.LiteralType).value;
      }

      if (type.flags & ctx.ts.TypeFlags.EnumLiteral)
        // FIXME 'nameType' is just some random name from generated Kotlin TypeObjectProperty.
        // FIXME This field should have its own name.
        tscType.nameType = getEnumQualifiedName(type, ctx);

      if (type.flags & ctx.ts.TypeFlags.TemplateLiteral)
        tscType.texts = (type as ts.TemplateLiteralType).texts;


      // FIXME internal API
      if (type.flags & ctx.ts.TypeFlags.TypeParameter && (type as any).isThisType)
        tscType.isThisType = true;

      // FIXME internal API
      if (typeof (type as any).intrinsicName === 'string')
        tscType.intrinsicName = (type as any).intrinsicName;


      let typeId = type.id
      tscType.id = typeId
      if (typeId) {
        ctx.checker.webStormCacheInfo?.seenTypeIds?.set(typeId, type)
      }

      return tscType;
    },
    ctx,
  );
}

function getEnumQualifiedName(type: ts.Type, ctx: ConvertContext): string | undefined {
  let qName = ''
  // FIXME internal API. Maybe use Node.parent instead?
  let current = (type.symbol as any).parent as ts.Symbol | undefined;
  while (current && !(current.valueDeclaration && ctx.ts.isSourceFile(current.valueDeclaration))) {
    qName = current.escapedName + (qName ? '.' + qName : '');
    current = (current as any).parent;
  }
  return qName || undefined;
}

function convertSymbol(
  symbol: ts.Symbol,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    symbol,
    ideObjectId => {
      const tscSymbol: Record<string, unknown> = {
        ideObjectId,
        ideObjectType: "SymbolObject",
        flags: symbol.flags,
        escapedName: symbol.escapedName as string,
      }

      tscSymbol.declarations = symbol.declarations
        ?.map(d => convertNode(d, ctx))
        ?.filter(isNotNull)

      if (symbol.valueDeclaration)
        tscSymbol.valueDeclaration = convertNode(symbol.valueDeclaration, ctx);

      // FIXME internal API
      if ((symbol as any).links?.type)
        tscSymbol.type = convertType((symbol as any).links?.type, ctx); // TS 5
      else if ((symbol as any).type)
        tscSymbol.type = convertType((symbol as any).type, ctx); // TS 4

      const symbolId = (ctx.ts as any).getSymbolId(symbol)
      ctx.checker.webStormCacheInfo?.seenSymbolIds?.set(symbolId, symbol)
      tscSymbol.id = symbolId

      return tscSymbol;
    },
    ctx,
    false, // To always have Symbol.escapedName
  );
}

function convertSignature(
  signature: ts.Signature,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    signature,
    ideObjectId => ({
      ideObjectId,
      ideObjectType: "SignatureObject",
      parameters: signature.parameters
        .map(s => convertSymbol(s, ctx))
        .filter(isNotNull),
      resolvedReturnType: convertType(ctx.checker.getReturnTypeOfSignature(signature), ctx),
    }),
    ctx,
  )
}

function convertNode(
  node: ts.Node,
  ctx: ConvertContext,
  childReverseMapping: any = undefined,
): object | undefined {
  return findReferenceOrConvert(
    node,
    ideObjectId => {
      if (ctx.ts.isSourceFile(node)) {
        return {
          ideObjectId,
          ideObjectType: "SourceFileObject",
          fileName: childReverseMapping?.fileName ?? node.fileName,
        };
      }
      else {
        const sourceFileParent = getSourceFileParent(node, ctx);
        const reverseMapping = runReverseMapper(sourceFileParent, node, ctx);

        return {
          ideObjectId,
          ideObjectType: "NodeObject",
          pos: reverseMapping?.pos ?? node.pos,
          end: reverseMapping?.end ?? node.end,
          parent: sourceFileParent
            ? convertNode(sourceFileParent, ctx, reverseMapping)
            : undefined,
        };
      }
    },
    ctx,
    false,  // To always have parent with fileName
  );
}

function getSourceFileParent(node: ts.Node, ctx: ConvertContext): ts.SourceFile | undefined {
  if (ctx.ts.isSourceFile(node)) return undefined;
  let current = node.parent;
  while (current) {
    if (ctx.ts.isSourceFile(current)) return current;
    current = current.parent;
  }
}

function runReverseMapper(sourceFileParent: ts.SourceFile | undefined, node: ts.Node, ctx: ConvertContext) {
  return ctx.reverseMapper && sourceFileParent
    ? ctx.reverseMapper(
      sourceFileParent,
      {
        start: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, node.pos),
        end: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, node.end),
      },
    )
    : undefined
}

function convertPseudoBigInt(
  pseudoBigInt: ts.PseudoBigInt,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    pseudoBigInt,
    ideObjectId => ({
      ideObjectId,
      ...pseudoBigInt,
    }),
    ctx,
    false, // For type safety
  );
}

function convertIndexInfo(
  indexInfo: object,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    indexInfo,
    ideObjectId => ({ideObjectId,}),
    ctx,
    false, // Objects are empty, no problem
  );
}

function findReferenceOrConvert(
  sourceObj: object,
  convertTarget: (ideObjectId: number) => object,
  ctx: ConvertContext,
  respectLevelLimitation = true,
): object | undefined {
  if (ctx.createdObjectsIdeIds.has(sourceObj)) {
    return {ideObjectIdRef: ctx.createdObjectsIdeIds.get(sourceObj)};
  }

  /*
   * Even with endless memory, level limitation needed because some generic symbols
   * always produce the new type. For example, Array.concat(): T[] will produce
   * another generic Array, not initial array. The property 'concat' of that generic Array
   * will produce yet another Array etc. Endless.
   */
  // TODO registry -- decrease for a huge project
  if (respectLevelLimitation && ctx.level >= 7) return undefined;

  const ideObjectId = ctx.nextId++;
  ctx.createdObjectsIdeIds.set(sourceObj, ideObjectId);

  ctx.level++;
  const newObject = convertTarget(ideObjectId);
  ctx.level--;

  return newObject;
}


function isNotNull<T>(t: T | undefined): t is T {
  return t != null
}

export function getSymbolType(
  ts: TS,
  program: ts.Program,
  symbolId: number,
  reverseMapper?: ReverseMapper,
): GetSymbolTypeResponse {
  const typeChecker = program.getTypeChecker()
  const cacheInfo = typeChecker.webStormCacheInfo
  if (!cacheInfo) {
    return undefined
  }
  let symbol = cacheInfo.seenSymbolIds.get(symbolId)
  if (!symbol) {
    return undefined
  }

  const ctx: ConvertContext = {
    ts: ts,
    checker: typeChecker,
    createdObjectsIdeIds: new Map(),
    reverseMapper,
    level: 0,
    nextId: 0,
  };
  let prepared: Record<string, unknown> = {}
  if ((ctx.checker as any).getTypeOfSymbol) {
    prepared = convertType((ctx.checker as any).getTypeOfSymbol(symbol), ctx) as Record<string, unknown>
  }
  prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;

  return {responseRequired: true, response: prepared}
}

export function getTypeProperties(
  ts: TS,
  program: ts.Program,
  typeId: number,
  reverseMapper?: ReverseMapper,
): GetElementTypeResponse {
  const typeChecker = program.getTypeChecker()
  const cacheInfo = typeChecker.webStormCacheInfo
  if (!cacheInfo) {
    return undefined
  }

  let type = cacheInfo.seenTypeIds.get(typeId)
  if (!type) {
    return undefined
  }

  const ctx: ConvertContext = {
    ts: ts,
    checker: typeChecker,
    createdObjectsIdeIds: new Map(),
    reverseMapper,
    level: 0,
    nextId: 0,
  }

  let prepared = convertTypeProperties(type, ctx) as Record<string, any>
  prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId

  return {responseRequired: true, response: prepared}
}

function convertTypeProperties(type: ts.Type, ctx: ConvertContext): object | undefined {
  return findReferenceOrConvert(type, ideObjectId => {
    let prepared: Record<string, unknown> = {
      ideObjectId,
      ideObjectType: "TypeObject",
      flags: type.flags,
      objectFlags: (type as ts.ObjectType).objectFlags
    }

    if (type.flags & ctx.ts.TypeFlags.Object) {
      assignObjectTypeProperties(type as ts.ObjectType, ctx, prepared)
    }
    if (type.flags & ctx.ts.TypeFlags.UnionOrIntersection) {
      assignUnionOrIntersectionTypeProperties(type as ts.UnionOrIntersectionType, ctx, prepared)
    }
    return prepared
  }, ctx, false)
}

function assignObjectTypeProperties(type: ts.ObjectType, ctx: ConvertContext, tscType: Record<string, unknown>) {
  tscType.constructSignatures = type.getConstructSignatures()
    .map((s: ts.Signature) => convertSignature(s, ctx))
    .filter(isNotNull);
  tscType.callSignatures = type.getCallSignatures()
    .map((s: ts.Signature) => convertSignature(s, ctx))
    .filter(isNotNull);
  tscType.properties = type.getProperties()
    .map((p: ts.Symbol) => convertSymbol(p, ctx))
    .filter(isNotNull);
  tscType.indexInfos = ctx.checker.getIndexInfosOfType &&
    ctx.checker.getIndexInfosOfType(type)
      .map((info: object) => convertIndexInfo(info, ctx))
      .filter(isNotNull);
}

function assignUnionOrIntersectionTypeProperties(type: ts.UnionOrIntersectionType, ctx: ConvertContext, tscType: Record<string, unknown>) {
  tscType.resolvedProperties = ctx.checker.getPropertiesOfType(type)
    .map((p: ts.Symbol) => convertSymbol(p, ctx))
    .filter(isNotNull);
}