import type * as ts from "typescript/lib/tsserverlibrary"
import type {
  GetElementTypeArguments,
  GetElementTypeResponse,
  GetSymbolTypeArguments,
  GetSymbolTypeResponse,
  GetTypePropertiesArguments
} from "./protocol"
import {getElementType, getSymbolType, getTypeProperties} from "./ide-get-element-type"

export function getElementTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                       projectService: ts.server.ProjectService,
                                       request: ts.server.protocol.Request): GetElementTypeResponse {
  const requestArguments = request.arguments as GetElementTypeArguments

  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let project = projectService.getDefaultProjectForFile(fileName, true)
  if (!project) {
    return undefined
  }

  // see getQuickInfoAtPosition
  let program = project.getLanguageService().getProgram();
  if (!program) {
    return undefined
  }
  const sourceFile = program.getSourceFile(fileName);
  if (!sourceFile) {
    return undefined
  }

  return getElementType(ts, program, sourceFile, requestArguments.range, requestArguments.forceReturnType)
}

export function getSymbolTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                      projectService: ts.server.ProjectService,
                                      request: ts.server.protocol.Request): GetSymbolTypeResponse {
  const requestArguments = request.arguments as GetSymbolTypeArguments

  const program = findProgram(projectService, requestArguments.ideTypeCheckerId)
  if (!program) {
    return undefined
  }
  return getSymbolType(ts, program, requestArguments.symbolId)
}

export function getTypePropertiesTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                          projectService: ts.server.ProjectService,
                                          request: ts.server.protocol.Request): GetElementTypeResponse {
  const requestArguments = request.arguments as GetTypePropertiesArguments

  const program = findProgram(projectService, requestArguments.ideTypeCheckerId)
  if (!program) {
    return undefined
  }
  return getTypeProperties(ts, program, requestArguments.typeId)
}


function findProgram(projectService: ts.server.ProjectService, ideTypeCheckerId: number): ts.Program | undefined {
  let configuredProjects = projectService.configuredProjects.values()
  while (true) {
    let next = configuredProjects.next()
    if (next.done) {
      break
    }
    else {
      let program = next.value.getLanguageService().getProgram()
      if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
        return program
      }
    }
  }

  for (let inferredProject of projectService.inferredProjects) {
    let program = inferredProject.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return program
    }
  }

  for (let externalProject of projectService.externalProjects) {
    let program = externalProject.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return program
    }
  }
}

