import gdb
import gdb.types

from default.printers import gdb_major_version

class CidrMiListFrameVariables(gdb.MICommand):
    """Lists variables in the current thread and frame"""

    frame_var_address_classes = {
        gdb.SYMBOL_LOC_ARG,
        gdb.SYMBOL_LOC_REF_ARG,
        gdb.SYMBOL_LOC_REGPARM_ADDR,
        gdb.SYMBOL_LOC_LOCAL,
        gdb.SYMBOL_LOC_STATIC,
        gdb.SYMBOL_LOC_REGISTER,
        gdb.SYMBOL_LOC_COMPUTED,
    }


    def __init__(self):
        super(CidrMiListFrameVariables, self).__init__("-cidr-stack-list-frame-variables")

    def invoke(self, argv):
        # the first argument == '1' means filter out-of-scope variables
        # if the argument is not present, filtering is disabled
        filter = False
        if len(argv) > 0:
            filter = argv[0] == '1'

        frame = gdb.selected_frame()
        if not frame.is_valid():
            return {'variables' : [], 'filtered' : False}

        result = []
        try:
            block = frame.block()
        except RuntimeError as e: # raised when block is not found
            block = None

        if filter:
            try:
                sal = frame.find_sal()
                if sal.symtab is None:
                    filter = False
                else:
                    frame_line = sal.line
                    frame_file = sal.symtab.filename
            except gdb.error as e:
                filter = False

        # to check standard gdb/mi -stack-list-frame-variables see list_args_or_locals in mi-cmd-stack.c
        # clashing variable names are handled by the client
        has_unknown_lines = False
        while block is not None and block.is_valid():
            for sym in block:
                if not sym.is_valid():
                    continue
                if sym.addr_class not in CidrMiListFrameVariables.frame_var_address_classes:
                    continue
                if not sym.is_variable and not sym.is_argument:
                    continue
                if filter:
                    if sym.line >= frame_line:
                        continue
                    symtab = sym.symtab
                    if symtab is None or symtab.filename != frame_file:
                        continue
                    # Line is usually 1-based, it is 0 when the corresponding
                    # dwarf attribute is missing (see new_symbol() in gdb/dwarf2/read.c).
                    # We check for 0 line after the filename, so that missing lines
                    # in other files don't affect the 'filtered' attribute in the response.
                    if sym.line == 0:
                        has_unknown_lines = True
                var = {'name' : sym.name}
                result.append(var)
            if block.function is not None:
                break
            block = block.superblock

        return {'variables' : result, 'filtered' : filter and not has_unknown_lines}


class CidrMiVarCreate(gdb.MICommand):
    """
    Wrapper around gdb/mi -var-create command.
    Uses the gdb.execute_mi api which was added in gdb 14.
    """

    _run = None

    def __init__(self):
        super(CidrMiVarCreate, self).__init__("-cidr-var-create")

    def invoke(self, argv):
        try:
            return CidrMiVarCreate._run(lambda: gdb.execute_mi("-var-create", *argv))
        except Exception as e:
            # wrap into gdb.GdbError to drop the 'Error occurred in Python:' prefix in error messages
            # and to disable python traceback printing
            # https://sourceware.org/gdb/current/onlinedocs/gdb.html/Exception-Handling.html#Exception-Handling
            raise gdb.GdbError(str(e))



# note that register_commands will fail for old gdb versions not supporting gdb.MICommand
# it is responsibility of the caller to check the version
def register_commands():
    CidrMiListFrameVariables()
    try:
        from libstdcxx.v6.printers import cidr_run_with_omitting_types_inside_to_string as run
    except ImportError:
        # libstdcxx doesn't support omitting types in to_string
        run = lambda fn: fn()
    CidrMiVarCreate._run = run
    CidrMiVarCreate()
