# Copyright 2000-2018 JetBrains s.r.o.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import gdb

import re


from libstdcxx.v6.printers import (Iterator,
                                   unique_ptr_get)


class StdChronoDurationPrinter:
    orig_to_string = None

    def to_string(self):
        s = StdChronoDurationPrinter.orig_to_string(self)
        return strip(s, 'std::chrono::duration = {', '}')


class StdChronoTimePointPrinter:
    prefix = re.compile(r'std::chrono::(?:\w+|[^{}=]+ time_point) = \{')
    orig_to_string = None

    def to_string(self, abbrev = None):
        if abbrev is not None:
            return StdChronoTimePointPrinter.orig_to_string(self, abbrev)
        s = StdChronoTimePointPrinter.orig_to_string(self)
        return strip(s, StdChronoTimePointPrinter.prefix, '}')


class StdChronoZonedTimePrinter:
    orig_to_string = None

    def to_string(self):
        s = StdChronoZonedTimePrinter.orig_to_string(self)
        return strip(s, 'std::chrono::zoned_time = {', '}')


class StdChronoTimeZonePrinter:
    orig_to_string = None

    def to_string(self):
        s = StdChronoTimeZonePrinter.orig_to_string(self)
        return strip(strip(s, self.typename), '=')


class StdChronoLeapSecondPrinter:
    orig_to_string = None

    def to_string(self):
        s = StdChronoLeapSecondPrinter.orig_to_string(self)
        return strip(strip(s, self.typename), '=')


class StdChronoTzdbPrinter:
    orig_to_string = None

    def to_string(self):
        s = StdChronoTzdbPrinter.orig_to_string(self)
        return strip(strip(s, self.typename), '=')


def strip(s, prefix, suffix=''):
    if isinstance(prefix, re.Pattern):
        match = prefix.match(s)
        start_matches = match is not None
        if start_matches:
            prefix = match.group()
    else:
        start_matches = s.startswith(prefix)

    if start_matches and s.endswith(suffix):
        return s[len(prefix):len(s)-len(suffix)].strip()
    return s


def patch_libstdcxx_printers_module():
    from libstdcxx.v6 import printers

    for patched_type in (StdChronoDurationPrinter,
                         StdChronoTimePointPrinter,
                         StdChronoZonedTimePrinter,
                         StdChronoTimeZonePrinter,
                         StdChronoLeapSecondPrinter,
                         StdChronoTzdbPrinter):
        name = patched_type.__name__
        orig_type = getattr(printers, name, None)
        if orig_type is not None:
            patched_type.orig_to_string = orig_type.to_string
            orig_type.to_string = patched_type.to_string
