import type * as ts from "typescript/lib/tsserverlibrary"
import type {
  GetElementTypeArguments,
  GetElementTypeResponse,
  GetSymbolTypeArguments,
  GetSymbolTypeResponse,
  GetTypePropertiesArguments
} from "./protocol"
import {getElementTypeByOffsets, getSymbolType, getTypeProperties} from "./ide-get-element-type"

const decoratedLanguageServices = new WeakSet();

export function decorateLanguageService(languageService: ts.LanguageService) {
  if (decoratedLanguageServices.has(languageService)) {
    return
  }
  decoratedLanguageServices.add(languageService);

  languageService.webStormGetElementType = (ts, fileName, startOffset, endOffset, forceReturnType, reverseMapper) => {
    // see getQuickInfoAtPosition
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined
    }

    return getElementTypeByOffsets(ts, program, sourceFile, startOffset, endOffset, forceReturnType, reverseMapper)
  }

  languageService.webStormGetSymbolType = (ts, symbolId, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getSymbolType(ts, program, symbolId, reverseMapper)
  }

  languageService.webStormGetTypeProperties = (ts, typeId, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeProperties(ts, program, typeId, reverseMapper)
  }
}

export function getElementTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                       projectService: ts.server.ProjectService,
                                       request: ts.server.protocol.Request): GetElementTypeResponse {
  const requestArguments = request.arguments as GetElementTypeArguments

  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let evalLocationPath = ts.server.toNormalizedPath(requestArguments.evalLocation)
  let project = projectService.getDefaultProjectForFile(evalLocationPath, true)
  if (!project) {
    return undefined
  }
  let range = requestArguments.range
  let sourceFile = project.getLanguageService(true).getProgram()?.getSourceFile(fileName)
  if (!sourceFile)
    return undefined

  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  return project.getLanguageService().webStormGetElementType(ts, fileName, startOffset, endOffset, requestArguments.forceReturnType)
}

export function getSymbolTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                      projectService: ts.server.ProjectService,
                                      request: ts.server.protocol.Request): GetSymbolTypeResponse {
  const requestArguments = request.arguments as GetSymbolTypeArguments

  const languageService = findLanguageService(projectService, requestArguments.ideTypeCheckerId)
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetSymbolType(ts, requestArguments.symbolId)
}

export function getTypePropertiesTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                          projectService: ts.server.ProjectService,
                                          request: ts.server.protocol.Request): GetElementTypeResponse {
  const requestArguments = request.arguments as GetTypePropertiesArguments

  const languageService = findLanguageService(projectService, requestArguments.ideTypeCheckerId)
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetTypeProperties(ts, requestArguments.typeId)
}


function findLanguageService(projectService: ts.server.ProjectService, ideTypeCheckerId: number): ts.LanguageService | undefined {
  let configuredProjects = projectService.configuredProjects.values()
  while (true) {
    let next = configuredProjects.next()
    if (next.done) {
      break
    }
    else {
      let program = next.value.getLanguageService().getProgram()
      if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
        return next.value.getLanguageService()
      }
    }
  }

  for (let inferredProject of projectService.inferredProjects) {
    let program = inferredProject.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return inferredProject.getLanguageService()
    }
  }

  for (let externalProject of projectService.externalProjects) {
    let program = externalProject.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return externalProject.getLanguageService()
    }
  }
}

